package arimaa3;

import ai_util.*;

import java.util.*;

/**
 * Base class to represent moves
 * @author hroch
 *
 */
public class ArimaaMove extends ArimaaBaseClass implements Comparable<Object> {

	/**
	 * TODO revert back to color related move represented with player
	 * perspective [0] represents player, [1] enemy! be carefull piece_bb[1][0]
	 * contains from_bb for 1 step (slide) move!! to allow isSlideOrDragLegal
	 * capture_mask and zorb_hash from gamestate perspective!!
	 */
	//public long piece_bb[][] = new long[2][]; // [2][GameState.Strongest+1] TODO replace !
	public long from_bb[]; // [GameState.Strongest+1]
	public long enemy_from_bb;
	public long to_bb[]; // [GameState.Strongest+1]
	public long enemy_to_bb;
	public int steps;
	public int move_ordering_value;
	public long zobr_hash;
	public long info_mask; 
	public static final long RAND_SCORE_MASK		= 0x00000000F0000000L;
	public static final long SETUP_MARK				= 0x8000000000000000L;
	public static final long ENEMY_MARK				= 0x4000000000000000L;
	// move contains dragging
	public static final long SILVER_MARK			= 0x0800000000000000L;
	// it's silvers turn
	public static final long PASS_MARK				= 0x0400000000000000L;
	// step (up to 3) is pass
	public static final long CAPTURE_COUNTS			= 0x0011333700113337L; 
	public static final long[] PLAYER_CAPTURE_COUNTS = {0x0000000000113337L, 0x0011333700000000L};
	// for more generalized rules it could be up to 0x7777777777777777L
	
	public ArimaaMove() {
		init();
	}

	public int compareTo(Object o) {
		ArimaaMove that = (ArimaaMove) o;
		return that.move_ordering_value - this.move_ordering_value;
	}

	/**
	 * Inits the move
	 */
	public void init() {
		from_bb = new long[GameState.Strongest + 1];
		to_bb = new long[GameState.Strongest + 1];
		//for (int p = 0; p < 2; p++) {
		//	piece_bb[p] = new long[GameState.Strongest + 1];
		//}
	}

	/**
	 * Clears the move
	 */
	public void clear() {
		enemy_from_bb=enemy_to_bb=0L; // minimizing it to consider more moves to be equal
		for (int s = 0; s <= GameState.Strongest; s++) {
			from_bb[s] = to_bb[s] = 0L;
		}
		steps = 0;
		move_ordering_value = 0;
		zobr_hash = 0L;
		long rand_score=0;//(GameState.random.nextInt()%9)-4;
		info_mask = (rand_score << 28) & RAND_SCORE_MASK;
	}

	/* Sums two subturns. 
	 * Cannot be used to sum full turns (or turns when the active player differs)
	 */
	public void add(ArimaaMove one, ArimaaMove two) {
		// this must differ from one and from two!
		// pass should be zero element ... so not changing colors
		if ((this==one)||(this==two)) {
			LogFile.message("Forbidden use of move add");
		}
		clear();// we use sum of cleared move with first step ... which should result in the first step ... so info_mask from two is deciding
		info_mask = (one.info_mask & CAPTURE_COUNTS)
				+ (two.info_mask & CAPTURE_COUNTS );
		info_mask |= (one.info_mask | two.info_mask) & (ENEMY_MARK | SILVER_MARK | RAND_SCORE_MASK);
		long temp_bb = one.enemy_to_bb & two.enemy_from_bb; // flip would give flipper to enemy otherwise 
		enemy_from_bb = (one.enemy_from_bb | two.enemy_from_bb) ^ temp_bb;
		enemy_to_bb = (one.enemy_to_bb | two.enemy_to_bb) ^ temp_bb;
		long same_color_bb = ~(enemy_from_bb ^ enemy_to_bb);
		for (int s = 0; s <= GameState.Strongest; s++) {
			if ((from_bb[s]=one.from_bb[s]|two.from_bb[s])!=0L) {
				temp_bb = one.to_bb[s]&two.from_bb[s];// push away pull another would stole dragged otherwise
				to_bb[s]=(one.to_bb[s]|two.to_bb[s])^temp_bb;;
				from_bb[s]=one.from_bb[s]|(two.from_bb[s]^temp_bb);
				if ((~(temp_bb = ~(from_bb[s]&to_bb[s]&same_color_bb)))!=0) {//TODO to je spatne!
					// this is not for piece moving twice, but for piece type returning to the same position
					to_bb[s]&=temp_bb;
					from_bb[s]&=temp_bb;
					enemy_from_bb&=temp_bb;
					enemy_to_bb&=temp_bb;
				}
			}
		}
		steps = one.steps + two.steps;
		move_ordering_value = one.move_ordering_value + two.move_ordering_value;
		zobr_hash = one.zobr_hash ^ two.zobr_hash;
	}

	/* gs gamestate prior the move
	 * returns true iff the move could be reverted in 1 or 2 steps
	 */
	public boolean fasterRevertExists(GameState gs) {
		if ((info_mask & CAPTURE_COUNTS) != 0) {return false;}
		long move_from_bb=getFromBits();
		if (enemy_from_bb==0) {// at least one opponent's piece must be moved
			return false;
		}
		int num_from_bits=Util.LowPopCnt(move_from_bb);
		if (num_from_bits>2) {return false;}
		if ((gs.frozen_pieces_bb & enemy_to_bb) != 0) {return false;}
		long move_to_bb=getToBits();
		if (num_from_bits==1) {
			int ind = Util.FirstOne(move_to_bb);
			int strength = gs.piece_at[ind]>>1;
			long back_bb=touching_bb(move_to_bb);
			if ((move_from_bb&back_bb)==0) {
				// the move squares must be touching ... not really diagonal flip or push one pull another could be reverted in 2 steps
				long middle_bb = touching_bb(move_from_bb)&touching_bb(move_to_bb)&~gs.player_bb[gs.enemy];
				// two steps to return back through the middle_bb
				if (middle_bb==0) {// the distance must be 2 (if not 1) and the path cannot be blocked by enemy
					return false;
				}
				//
				// from Er.M to ErrM is revertible but through another path (exchanging stones) 
				//      .r..    ....
				// ------
				//        
				//      Er.. to Err. is revertible only through the original path
				//       r.M       M
				// ------
				//      Er. to E.. is revertible exchanging stones
				//     c r    crr.
				if ((middle_bb & gs.empty_bb & 
					(gs.count_touch_bb[gs.player][2] | ~touching_bb(gs.stronger_or_eq_bb[strength+1])))!=0) {
					if (strength>0) {
						return true; // empty path friendly or without freezer 
					} else {
						long stepDests_bb=(gs.empty_bb&middle_bb)|move_from_bb,stepSources_bb=move_to_bb|(gs.empty_bb&middle_bb);
						if ((can_step_bb(stepSources_bb, gs.player)&stepDests_bb)==stepDests_bb) {
							return true; // empty path friendly or without freezer which can the rabbit run
						}
					}
				}
				if ((middle_bb&gs.stronger_or_eq_bb[strength]^gs.stronger_or_eq_bb[strength+1]^gs.player_bb[gs.player])!=0) {
					// the same piece on the way back could the 2nd step be made?
					// we know the dragger touches move_to_bb so it would be frozen unless it has another friend
					if ((gs.count_touch_bb[gs.player][2]&move_to_bb)!=0) {
						if (strength>0) {
							return true; // both pieces could step back once
						} else {
							long stepDests_bb=(middle_bb&gs.player_bb[gs.player])|move_from_bb,
								stepSources_bb=move_to_bb|(middle_bb&gs.player_bb[gs.player]);
							if ((can_step_bb(stepSources_bb, gs.player)&stepDests_bb)==stepDests_bb) {
								return true;
							}
						}
					} else {
						return false;
					}
				}
				return false;
			} else {
				// "slide" from move_from_bb to move_to_bb (piece is not frozen)
				if (strength>0) {// piece could just step back
					return true;
				} else {
					// could enemy rabbit step back?
					return (can_step_bb(move_to_bb, gs.player) & move_from_bb)!=0;
				}
			}
		} else {// move_from_bb has 2 bits either are both from player(original enemy) ... push pull
			// or one from player, one from enemy
			long middle_bb = move_from_bb & move_to_bb;
			if (middle_bb == 0) {
				return false;
			}
			if (move_from_bb == enemy_from_bb) {// push pull 
				// Eh ... Emh  so it looks like push by enemies own piece 
				//  m
				int ind = Util.FirstOne(middle_bb);
				int strength = gs.piece_at[ind]>>1;
				if (strength == 0) {// rabbits retreat possibility should be tested
					if ((can_step_bb(middle_bb, gs.player) & move_from_bb)==0) {
						return false;
					}
				}
				// first piece was not frozen, but the second step must be checked
				ind = Util.FirstOne(move_to_bb^middle_bb);
				strength = gs.piece_at[ind]>>1;
				if (((move_to_bb^middle_bb)&(gs.count_touch_bb[gs.player][2]|~(gs.dominated_pieces_bb)))!=0) {
					if (strength>0) {
						return true; // both pieces could step back once
					} else {// rabbits return must be checked
						return (can_step_bb(move_to_bb, gs.player)&middle_bb)==middle_bb;
					}
				} else {// frozen
					return false;
				}				
			} else {// drag, return and step ... return forces push so push return and step
				// step must be to enemy_from_bb or from enemy_to_bb so it looks like drag with weaker piece
				// Em ... EHm  EmH  E mH
				//  H            R    R
				// so redragger piece should be unfrozen and stronger to replacing piece
				//TODO
				int ind = Util.FirstOne(move_to_bb & ~enemy_to_bb);
				int strength = gs.piece_at[ind]>>1;// the piece to be redragged
				long move_bb = move_from_bb|move_to_bb; 
				if ((move_bb&touching_bb(middle_bb))!=(move_bb^middle_bb)) {// drag requires neighbours
					return false;
				}
				return (move_to_bb&~gs.frozen_pieces_bb&gs.stronger_or_eq_bb[strength+1])!=0;
			}
		}
	}
	
	public void copy(ArimaaMove one) {// could copy itself
		for (int s = 0; s <= GameState.Strongest; s++) {
			from_bb[s]=one.from_bb[s];
			to_bb[s]=one.to_bb[s];
			//for (int p = 0; p < 2; p++) {
			//	piece_bb[p][s] = one.piece_bb[p][s];
			//}
		}
		enemy_from_bb=one.enemy_from_bb;
		enemy_to_bb=one.enemy_to_bb;
		steps = one.steps;
		move_ordering_value = one.move_ordering_value;
		zobr_hash = one.zobr_hash;
		info_mask = one.info_mask;
	}

	/**
	 * Takes the two game states and returns the move to get from start to end.
	 * @param start GameState
	 * @param end GameState
	 */
	public void difference(GameState start, GameState end) {
		init(); clear(); // generates random offset
		for (int c = 0; c < 2; c++) {
			for (int s = 0; s <= GameState.Strongest; s++) {
				// PopCnt optimization ... rabbits separately
				long start_piece_bb, end_piece_bb;
				if (0 != ((start_piece_bb = start.player_bb[c]&(start.stronger_or_eq_bb[s]
						^ start.stronger_or_eq_bb[s + 1]))
						^ (end_piece_bb = end.player_bb[c]&(end.stronger_or_eq_bb[s]
								^ end.stronger_or_eq_bb[s + 1])))) {
					// changed in this color strength
					long stay_bb = start_piece_bb & end_piece_bb;
					start_piece_bb &= ~stay_bb;
					end_piece_bb &= ~stay_bb;
					from_bb[s] |= start_piece_bb;
					to_bb[s] |= end_piece_bb;
					if (c != start.player) {
						enemy_from_bb |= start_piece_bb;
						enemy_to_bb |= end_piece_bb;
						info_mask |= ENEMY_MARK;
					}
				}
				long captured = (s == 0) ? Util.PopCnt(start_piece_bb)
						- Util.PopCnt(end_piece_bb) : Util
						.LowPopCnt(start_piece_bb)
						- Util.LowPopCnt(end_piece_bb);
				if (captured > 0) {
					info_mask += (captured) << ((c << 5) + (s << 2));
				}
			}
		}
		if (start.player == PL_SILVER) {
			info_mask |= SILVER_MARK;
		}
		steps = start.steps_remaining - end.steps_remaining;
		move_ordering_value = 0;
		zobr_hash = start.zobr_hash ^ end.zobr_hash;
		if (steps<=0) {
			steps+=4;
		}
	}

	/**
	 * Returns all the set bits ie Any squares involved in the move
	 * 
	 * @return long
	 */
	public long getFromBits() {
		long result = 0;
		for (int s = 0; s <= GameState.Strongest; s++) {
			result |= from_bb[s];
		}
		return result;
	}

	/**
	 * Returns all the set bits ie Any squares involved in the move
	 * 
	 * @return long
	 */
	public long getToBits() {
		long result = 0;
		for (int s = 0; s <= GameState.Strongest; s++) {
			result |= to_bb[s];
		}
		return result;
	}

	/**
	 * Returns all the set bits ie Any squares involved in the move
	 * 
	 * @return long
	 */
	public long getSetBits() {
		return getFromBits() | getToBits();
	}

	public boolean isEmpty() {
		if (this.zobr_hash != 0) {
			return false;
		}
		if (enemy_from_bb != 0L) {
			return false;
		}
		for (int s = 0; s <= GameState.Strongest; s++) {
			if (from_bb[s] != 0L) {
				return false;
			}
		}
		return true;
	}

	public boolean equalsUptoSteps(Object o) {
		if (!(o instanceof ArimaaMove)) {
			return false;
		}
		ArimaaMove that = (ArimaaMove) o;

		if (this.zobr_hash != that.zobr_hash) {
			return false;
		}

		// equal with very high probability but check it
		if (this.enemy_from_bb != that.enemy_from_bb) {
			return false;
		}
		if (this.enemy_to_bb != that.enemy_to_bb) {
			return false;
		}
		for (int s = 0; s <= GameState.Strongest; s++) {
			if (this.from_bb[s] != that.from_bb[s]) {
				return false;
			}
			if (this.to_bb[s] != that.to_bb[s]) {
				return false;
			}
		}
		return true;
	}
	
	public boolean equals(Object o) {
		if (!(o instanceof ArimaaMove)) {
			return false;
		}
		ArimaaMove that = (ArimaaMove) o;

		if (this.steps != that.steps) {
			return false;
		}

		if (this.zobr_hash != that.zobr_hash) {
			return false;
		}

		// equal with very high probability but check it
		if (this.enemy_from_bb != that.enemy_from_bb) {
			return false;
		}
		if (this.enemy_to_bb != that.enemy_to_bb) {
			return false;
		}
		for (int s = 0; s <= GameState.Strongest; s++) {
			if (this.from_bb[s] != that.from_bb[s]) {
				return false;
			}
			if (this.to_bb[s] != that.to_bb[s]) {
				return false;
			}
		}
		return true;
	}
	/**
	 * Dumps out a one line text representation of the move This is NOT OFFICIAL
	 * ARIMAA Notation Prints out number of steps, then displays every set bit
	 * it is just for debugging so piece name color switch is not problem
	 * to_bb/from_bb in slides is a bonus
	 * 
	 * @return String
	 */

	public String toString() {
		String result = "{" + steps + "}";
		boolean is_pass_move = true;
		int player = (info_mask & SILVER_MARK)==0 ? PL_GOLD : PL_SILVER;
		int enemy=player ^ 1;
		for (int s = GameState.Strongest; s>=0 ; s--) {
			if (from_bb[s] != 0) {
				is_pass_move = false;
				long efrom_bb = from_bb[s] & enemy_from_bb;
				if (efrom_bb != 0) {
					String temp = GameState.StrengthAndColor2Name(s, enemy);
					long temp_bb = efrom_bb;
					while (temp_bb != 0) {
						int index = Util.FirstOne(temp_bb);
						temp_bb ^= (1L << index);
						temp += Notation.Index2Name(index);
					}
					temp += ">";
					temp_bb=to_bb[s] & enemy_to_bb;
					while (temp_bb != 0) {
						int index = Util.FirstOne(temp_bb);
						temp_bb ^= (1L << index);
						temp += Notation.Index2Name(index);
					}
					result += temp + " ";
				}
				long temp_bb = from_bb[s] ^ efrom_bb;
				if (temp_bb!=0) {
					String temp = GameState.StrengthAndColor2Name(s, player);					
					while (temp_bb != 0) {
						int index = Util.FirstOne(temp_bb);
						temp_bb ^= (1L << index);
						temp += Notation.Index2Name(index);
					}
					temp += ">";
					temp_bb=to_bb[s] & ~enemy_to_bb;
					while (temp_bb != 0) {
						int index = Util.FirstOne(temp_bb);
						temp_bb ^= (1L << index);
						temp += Notation.Index2Name(index);
					}
					result += temp + " ";
				}
			}
		}

		if (is_pass_move) {
			result += "pass";
		}
		//result+="(Z: "+zobr_hash+")(IM: "+Long.toHexString(info_mask)+")";
		return result;
	}

	/**
	 * Creates a move assuming move_text is official arimaa Notation must detect who is playing 
	 * the strongest piece included (first 2 steps would suffice for test) defines that
	 * @param move_text String
	 */
	public ArimaaMove(String move_text) {
		init();
		// Clear the move
		clear();
		int strongest[] = { -1, -1 };

		// Steps are always 4
		this.steps = 4;
		long color_from_bb[]={0,0}, color_to_bb[]={0,0};

		// Break the move into its component parts
		StringTokenizer tokenizer = new StringTokenizer(move_text);

		while (tokenizer.hasMoreTokens()) {
			String move = tokenizer.nextToken();

			if (move.equals("pass")) {
				break;
			}

			// Convert the text move to numbers
			int piece_type = GameState.Name2InclPiece(move.substring(0, 1));
			int c = GameState.InclPiece2Color(piece_type);
			int s = GameState.InclPiece2Strength(piece_type);

			assert (s >= 0 && s <= GameState.Strongest);
			int col = Notation.Name2Col(move.substring(1, 2));
			int row = Notation.Name2Row(move.substring(2, 3));
			int index = Notation.RowCol2Index(row, col);
			long f_ind_bb = 1L << index;
			long cont_bb = to_bb[s]&f_ind_bb;
			f_ind_bb ^= cont_bb;
			to_bb[s]^=cont_bb;
			color_to_bb[c]^=cont_bb;
			int direction = move.length()==3?5:Notation.Name2Dir(move.substring(3, 4));

			// Handles captures
			if (direction == 5) {
				info_mask |= SETUP_MARK;
				if (s > strongest[c]) {
					strongest[c] = s;
				}
				color_to_bb[c] ^= f_ind_bb;
				to_bb[s] ^= f_ind_bb;
				zobr_hash ^= GameState.Zobr[s][c][index];
				info_mask += 1L << ((c << 5) + (s << 2));
			}
			if (direction == 4) {
				color_from_bb[c] ^= f_ind_bb;
				from_bb[s] ^= f_ind_bb;
				zobr_hash ^= GameState.Zobr[s][c][index];
				info_mask += 1L << ((c << 5) + (s << 2));
			}
			if (direction < 4) {
				if (s > strongest[c]) {
					strongest[c] = s;
				}
				int dir_offset[] = { 8, -8, 1, -1, 0 };
				int to_index = index + dir_offset[direction];
				color_from_bb[c] ^= f_ind_bb;
				from_bb[s] ^= f_ind_bb;
				long t_ind_bb = 1L << to_index;
				color_to_bb[c] ^= t_ind_bb;
				to_bb[s] ^= t_ind_bb;
				long both_bb = (from_bb[s]&color_from_bb[c]) & (to_bb[s]&color_to_bb[c]);
				if (both_bb!=0) {
					from_bb[s]&=~both_bb;
					to_bb[s]&=~both_bb;
					color_from_bb[c]&=~both_bb;
					color_to_bb[c]&=~both_bb;
				}
				zobr_hash ^= GameState.Zobr[s][c][index]
						^ GameState.Zobr[s][c][to_index];
			}
		}
		if (strongest[PL_SILVER] > strongest[PL_GOLD]) {
			// switch to silver's perspective
			info_mask |= SILVER_MARK;
			enemy_from_bb = color_from_bb[PL_GOLD];
			enemy_to_bb = color_to_bb[PL_GOLD];
			if (strongest[PL_GOLD] >= 0) {
				info_mask |= ENEMY_MARK;
			}
		} else {
			enemy_from_bb = color_from_bb[PL_SILVER];
			enemy_to_bb = color_to_bb[PL_SILVER];
			if (strongest[PL_SILVER] >= 0) {
				info_mask |= ENEMY_MARK;
			}
		}
	}

	/**
	 * @param from_index
	 * @param to_index
	 * @return direction as a string (one letter)
	 */
	public String getStringDirection(int from_index, int to_index) {
		int dif = to_index - from_index;
		// 0xffff 11, 0x0001 10, 0xfff8 1, 0x0008 0
		return Notation.Dir2Name(((dif & 0x01) << 1) | ((dif & 0x10) >>> 4));
	}

	/**
	 * Converts a simple move to official Notation 
	 * A simple move is slide or drag = push/pull
	 * Nothing else.
	 */
	public String SlideOrDragToOfficialArimaaNotation() {
		String result = "";
		int player = (info_mask & SILVER_MARK)==0 ? PL_GOLD : PL_SILVER;
		if ((info_mask & PASS_MARK) != 0) {
			// Handle pass move
			return "pass";
		} else if (this.steps == 1) {
			// Handle slide moves
			String suicide = "", piece_name;
			for(int s=0;s<=GameState.Strongest;s++) {
				if (from_bb[s]!=0) {
					piece_name = GameState.StrengthAndColor2Name(s, player);
					if (atMostOneBitSet(from_bb[s])) {
						if (to_bb[s]==0) {
							int from_index = Util.bit2ind(from_bb[s]);
							if ((from_bb[s]&TRAP_SQUARES)!=0) {
								suicide += " " + piece_name + Notation.Index2Name(from_index)+"x";								
							} else {
								long sac_bb = touching_bb(from_bb[s]) & TRAP_SQUARES;
								int sac_index = Util.bit2ind(sac_bb);
								result += piece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, sac_index);
								suicide += " " + piece_name + Notation.Index2Name(sac_index)+"x";
							}
						} else {
							int from_index = Util.bit2ind(from_bb[s]);
							int to_index = Util.bit2ind(to_bb[s]);
							result += piece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, to_index);							
						}
					} else {
						long sac_bb = from_bb[s] & TRAP_SQUARES;
						int sac_index = Util.bit2ind(sac_bb);
						suicide += " " + piece_name + Notation.Index2Name(sac_index)+"x";
						int from_index = Util.bit2ind(from_bb[s] ^ sac_bb);
						int to_index = Util.bit2ind(to_bb[s]);
						result += piece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, to_index);							
					}
				}
			}
			result += suicide; suicide ="";
		} else if (this.steps == 2) {
			// Handle drag moves
			// This does *NOT* handle two consecutive step moves

			String player_step="", enemy_step="", player_suicide="", enemy_suicide = "";
			long efrom_bb, eto_bb, pfrom_bb, pto_bb;
			long estep_from_bb=0, pstep_to_bb=0;
			for(int s=0;s<=GameState.Strongest;s++) {
				if (from_bb[s]!=0) {
					if ((efrom_bb=(from_bb[s]&enemy_from_bb))!=0) {
						String epiece_name = GameState.StrengthAndColor2Name(s, player^1);
						if (atMostOneBitSet(efrom_bb)) {
							if ((eto_bb=(to_bb[s]&enemy_to_bb))==0) {
								int from_index = Util.bit2ind(efrom_bb);
								if ((efrom_bb&TRAP_SQUARES)!=0) {
									enemy_suicide += " " + epiece_name + Notation.Index2Name(from_index)+"x";								
								} else {
									long sac_bb = touching_bb(efrom_bb) & TRAP_SQUARES;
									int sac_index = Util.bit2ind(sac_bb);
									estep_from_bb = efrom_bb;
									enemy_step += epiece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, sac_index);
									enemy_suicide += " " + epiece_name + Notation.Index2Name(sac_index)+"x";
								}
							} else {
								int from_index = Util.bit2ind(efrom_bb);
								int to_index = Util.bit2ind(eto_bb);
								estep_from_bb = efrom_bb;
								enemy_step += epiece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, to_index);							
							}
						} else {
							long sac_bb = efrom_bb & TRAP_SQUARES;
							int sac_index = Util.bit2ind(sac_bb);
							enemy_suicide += " " + epiece_name + Notation.Index2Name(sac_index)+"x";
						}
					}
					if ((pfrom_bb=(from_bb[s]&~enemy_from_bb))!=0) {
						String ppiece_name = GameState.StrengthAndColor2Name(s, player);
						if (atMostOneBitSet(pfrom_bb)) {
							if ((pto_bb=(to_bb[s]&~enemy_to_bb))==0) {
								int from_index = Util.bit2ind(pfrom_bb);
								if ((pfrom_bb&TRAP_SQUARES)!=0) {
									player_suicide += " " + ppiece_name + Notation.Index2Name(from_index)+"x";								
								} else {
									long sac_bb = touching_bb(pfrom_bb) & TRAP_SQUARES;
									int sac_index = Util.bit2ind(sac_bb);
									pstep_to_bb = sac_bb;
									player_step += ppiece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, sac_index);
									player_suicide += " " + ppiece_name + Notation.Index2Name(sac_index)+"x";
								}
							} else {
								int from_index = Util.bit2ind(pfrom_bb);
								int to_index = Util.bit2ind(pto_bb);
								pstep_to_bb = pto_bb;
								player_step += ppiece_name + Notation.Index2Name(from_index) + getStringDirection(from_index, to_index);							
							}
						} else {
							long sac_bb = pfrom_bb & TRAP_SQUARES;
							int sac_index = Util.bit2ind(sac_bb);
							player_suicide += " " + ppiece_name + Notation.Index2Name(sac_index)+"x";
						}
					}
				}
			}
			if (estep_from_bb==pstep_to_bb) {
				result = enemy_step+enemy_suicide+" "+player_step+player_suicide;
			} else {
				result = player_step+player_suicide+" "+enemy_step+enemy_suicide;
			}
		} else {
			LogFile.message("Bug SlideOrDragToOfficialNotation too many steps:"+this);			
		}
		return result;
	}

	/**
	 * tests if given slide/drag could be played in current GameState it is
	 * granted it could be played on another state, so relative power of
	 * pieces need not be tested what must be tested: 1) player piece is not
	 * frozen 2) the empty space is where the first step went 3) the piece(s) to
	 * move are on correct place, pieces lost in trap stopped being supported
	 */
	public boolean isSlideOrDragLegal(GameState gs) {
		int player = (info_mask & SILVER_MARK)==0 ? PL_GOLD : PL_SILVER;
		if (this.steps == 1) {
			for(int s=0;s<=GameState.Strongest;s++) {
				if (from_bb[s]!=0) {
					if ((gs.player_bb[player]&(gs.stronger_or_eq_bb[s]^gs.stronger_or_eq_bb[s+1])&from_bb[s])!=from_bb[s]) {
						return false;
					}
					if (atMostOneBitSet(from_bb[s])) {
						if (to_bb[s]==0) {
							if ((from_bb[s]&TRAP_SQUARES)!=0) {
								if ((gs.suicide_bb[player] & from_bb[s])==0) {
									return false;
								}
							} else {
								long sac_bb = touching_bb(from_bb[s]) & TRAP_SQUARES;
								if ((gs.empty_bb & sac_bb & gs.suicide_bb[player])==0) {
									return false;
								}
								if ((gs.frozen_pieces_bb & from_bb[s])!=0) {
									return false;
								}
							}
						} else {
							if ((gs.frozen_pieces_bb & from_bb[s])!=0) {
								return false;
							}
							if ((gs.empty_bb & to_bb[s])==0) {
								return false;
							}
						}
					} else {//from_bb[s] were touching each other so none is frozen
						long sac_bb = from_bb[s] & TRAP_SQUARES;
						if ((sac_bb & gs.suicide_bb[player])==0) {
							return false;
						}
						if ((gs.empty_bb & to_bb[s])==0) {
							return false;
						}
					}
				}
			}
		} else if (this.steps == 2) {
			// Handle drag moves
			long efrom_bb, eto_bb, pfrom_bb, pto_bb;
			long estep_from_bb=0, estep_to_bb=0, pstep_to_bb=0;
			for(int s=0;s<=GameState.Strongest;s++) {
				if (from_bb[s]!=0) {
					if (((gs.stronger_or_eq_bb[s]^gs.stronger_or_eq_bb[s+1])&from_bb[s])!=from_bb[s]) {
						return false;
					}
					if ((efrom_bb=(from_bb[s]&enemy_from_bb))!=0) {
						if ((gs.player_bb[gs.enemy]&efrom_bb)!=efrom_bb) {
							return false;
						}
						if (atMostOneBitSet(efrom_bb)) {
							if ((eto_bb=(to_bb[s]&enemy_to_bb))==0) {
								if ((efrom_bb&TRAP_SQUARES)!=0) {
									if ((gs.suicide_bb[player^1] & efrom_bb)==0) {
										return false;
									}
								} else {
									long sac_bb = touching_bb(efrom_bb) & TRAP_SQUARES;
									estep_from_bb = efrom_bb;
									estep_to_bb = sac_bb;
									if ((sac_bb & gs.suicide_bb[player^1])==0) {
										return false;
									}
								}
							} else {
								estep_from_bb = efrom_bb;
								estep_to_bb = eto_bb;
							}
						} else {
							long sac_bb = efrom_bb & TRAP_SQUARES;
							if ((sac_bb & gs.suicide_bb[player^1])==0) {
								return false;
							}
							estep_from_bb = efrom_bb & ~TRAP_SQUARES;
							estep_to_bb = to_bb[s] & enemy_to_bb;
						}
					}
					if ((pfrom_bb=(from_bb[s]&~enemy_from_bb))!=0) {
						if ((gs.player_bb[gs.player]&pfrom_bb)!=pfrom_bb) {
							return false;
						}
						if (atMostOneBitSet(pfrom_bb)) {
							if ((pto_bb=(to_bb[s]&~enemy_to_bb))==0) {
								if ((pfrom_bb & TRAP_SQUARES)!=0) {
									if ((gs.suicide_bb[player] & pfrom_bb)==0) {
										return false;
									}
								} else {
									long sac_bb = touching_bb(pfrom_bb) & TRAP_SQUARES;
									if ((sac_bb & gs.suicide_bb[player])==0) {
										return false;
									}
									if ((gs.frozen_pieces_bb & pfrom_bb)!=0) {
										return false;
									}
									pstep_to_bb = sac_bb;
								}
							} else {
								if ((gs.frozen_pieces_bb & pfrom_bb)!=0) {
									return false;
								}
								pstep_to_bb = pto_bb;
							}
						} else {// the pieces are touching each other so no freeze possible
							long sac_bb = pfrom_bb & TRAP_SQUARES;
							if ((sac_bb & gs.suicide_bb[player])==0) {
								return false;
							}
							pstep_to_bb = to_bb[s]&~enemy_to_bb;
						}
					}
				}
			}
			if (estep_from_bb==pstep_to_bb) {
				if ((gs.empty_bb & estep_to_bb)==0) {
					return false;
				}
			} else {
				if ((gs.empty_bb & pstep_to_bb)==0) {
					return false;
				}
			}
		}
		return true;
	}
	
	/** gs gamestate prior the move
	 * returns true iff the 3 of 4 steps could be reverted in 1 step
	 */
	public boolean revert3In1Exists(GameState gs) {//TODO
		boolean todo = true;
		if (todo) return false;
		int num_steps=0;long efrom_bb, pfrom_bb=0, eto_bb, pto_bb=0;
		for(int s = GameState.Strongest; s >= 0; s--) {
			if (from_bb[s]!=0) {
				if ((efrom_bb=(from_bb[s]&enemy_from_bb))!=0) {
					num_steps+=1;
					if (num_steps>2) {
						return false;
					}
					if (!atMostOneBitSet(efrom_bb)) {
						return false;
					}
					if ((eto_bb=(to_bb[s]&enemy_to_bb))==0) {
						return false;
					}
					if ((can_step_bb(eto_bb,GameState.StrengthAndColor2InclPiece(s,gs.enemy))&efrom_bb)==0) {
						return false;
					}
					if ((gs.count_touch_bb[gs.enemy][2]&eto_bb)==0) {// will not have friends
						long frozers_bb = (gs.stronger_or_eq_bb[s+1]^(pfrom_bb|pto_bb));
						if ((frozers_bb&touching_bb(eto_bb))!=0) {
							return false;
						}
					}
				}
				if ((from_bb[s]&~enemy_from_bb)!=0) {
					pfrom_bb=from_bb[s]&~enemy_from_bb;
					num_steps+=1;
					if (num_steps>2) {
						return false;
					}
					if (!atMostOneBitSet(pfrom_bb)) {
						return false;
					}
					if ((pto_bb=(to_bb[s]&~enemy_to_bb))==0) {
						return false;
					}
					if ((touching_bb(pto_bb)&pfrom_bb)==0) {
						return false;
					}
				}
			}
		}
		return false;
	}
	
	public static void main(String args[]) {

		String tests[] = {
				"6|2b %13 +-----------------+%138| r r   c c r   r |%137|     r     d     |%136| r H             |%135|                 |%134|           E r   |%133| R H       R M d |%132|           e R r |%131|   R   R C h   R |%13 +-----------------+%13   a b c d e f g h%13", // Belbo
				"3|2w %13 +-----------------+%138|   r r d H r r r |%137| r     c E R   r |%136|   h             |%135|                 |%134|     m           |%133|       r         |%132|     M e   R     |%131| R     R D   R R |%13 +-----------------+%13   a b c d e f g h%13", 
				// 99of9 vs RonWeasley Test position
				"4|3w %13 +-----------------+%138| r r   e r r r r |%137|       r r   r   |%136|     X C   X     |%135|     R           |%134|                 |%133|     X     X     |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
				"8|4b %13 +-----------------+%138|     r r r r M r |%137| r         c r   |%136|       d d       |%135| h D c m         |%134|     E e         |%133|     r           |%132|   R h         H |%131| R R   H R R R R |%13 +-----------------+%13   a b c d e f g h%13",
				"15|5b %13 +-----------------+%138|         d r     |%137|   r     M r e   |%136|   d   c   R     |%135|       c   E   R |%134| r   H         r |%133| D h R   d   C   |%132| r D m   R   r C |%131| R R       R   R |%13 +-----------------+%13   a b c d e f g h%13",
				"5|6b %13 +-----------------+%138|   c   d     c   |%137|         d r     |%136|               C |%135|   r             |%134|             r h |%133|   M   R   r r R |%132| R h   m E e H   |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
				"19|7b %13 +-----------------+%138|         d r     |%137|   r     M r e   |%136|   d   c     R   |%135|       c   E   R |%134| r   H         r |%133| D h R   d   r C |%132| r D m   C       |%131| R R     R   R R |%13 +-----------------+%13   a b c d e f g h%13",
				"4|8w %13 +-----------------+%138|               H |%137| r         M e   |%136|   d   c     R   |%135|       c     E R |%134| r   H         r |%133| D h R   d   r C |%132| r D m         C |%131| R R     R   R R |%13 +-----------------+%13   a b c d e f g h%13",
				"0|9b %13 +-----------------+%138| r   r   r d r r |%137| d           R   |%136|       E e       |%135|       R   m     |%134|     D R     R r |%133|   R           R |%132| R               |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13",
				"0|10w %13 +-----------------+%138| r r r r r r r r |%137| R R R R R   C h |%136|         H   e M |%135|               R |%134|           E     |%133|             D   |%132|                 |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13",
				"9|11w %13 +-----------------+%138|                 |%137|   r             |%136|   h   E M   r   |%135|   D d     H r   |%134|       H   c     |%133|       C     C   |%132| R r     D     R |%131| R R R     R R R |%13 +-----------------+%13   a b c d e f g h%13", 
				// Game Number: 21399
				"0|12b %13 +-----------------+%138| r   r   r d r r |%137| d           R   |%136|       E e       |%135|     D R   m     |%134|         R   R r |%133|   R           R |%132| R               |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13",
				"0|13w %13 +-----------------+%138| r   r   r d r r |%137| d           R   |%136|       E e       |%135|       R   m     |%134|     D R R     r |%133|   R           R |%132| R               |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13",
				"0|14b %13 +-----------------+%138| r r         r r |%137| r   r     r   r |%136|                 |%135|     m E   R     |%134|       C         |%133|         M   D   |%132|     e     H     |%131| R R       R R R |%13 +-----------------+%13   a b c d e f g h%13",
				"0|15b %13 +-----------------+%138| r   r   r d r r |%137| d           R   |%136|       E e       |%135|     R     m     |%134|     D   R   R r |%133|   R           R |%132| R               |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13",
				"0|16b %13 +-----------------+%138| r r     r d r r |%137| d           R r |%136|       E e       |%135|       R   m     |%134|     D R R       |%133|   R           R |%132| R               |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13",
				"0|17b %13 +-----------------+%138| r r     r d r r |%137|   d         R   |%136|       E e       |%135|     R     m r   |%134|     D   R   R   |%133|   R           R |%132| R               |%131|   C             |%13 +-----------------+%13   a b c d e f g h%13", };
		GameState.Strongest = 5;
		MoveList step_list = new MoveList(false);

		for (String pos_text : tests) {
			int pos_start = pos_text.indexOf("|")+1;
			int exp_num = Integer.parseInt(pos_text.substring(0,pos_start-1));
			String text = pos_text.substring(pos_start);

			GameState gs = new GameState(text);
			step_list.clear();
			System.out.println(gs.toBoardString());
			gs.genDragMoves(step_list, FULL_BB, FULL_BB, FULL_BB, FULL_BB);
			if (step_list.levelSize()!=exp_num) {
				System.out.println("Bug report");
				System.out.println(gs);
				System.out.println("Expected: "+exp_num+" gained: " + step_list.levelSize());
			} else {
				System.out.println("OK "+exp_num+" drags found.");				
			}
			for (step_list.levelFirst();step_list.levelCont();step_list.levelNext()) {
				ArimaaMove step = step_list.getCurrentMove();
				System.out.println(step);
				System.out.println(step.SlideOrDragToOfficialArimaaNotation());
			}
			step_list.clear();
			gs.genSlideMoves(step_list, FULL_BB, FULL_BB);
			System.out.println("Slides:");
			for (step_list.levelFirst();step_list.levelCont();step_list.levelNext()) {
				ArimaaMove step = step_list.getCurrentMove();
				System.out.println(step);
				System.out.println(step.SlideOrDragToOfficialArimaaNotation());
			}
		}
		String text = "3w %13 +-----------------+%138| r r r d d r r r |%137| r   c m   c h r |%136|   h             |%135|       E         |%134|         e       |%133|         D       |%132| H H C     R M D |%131| R R R R R R C R |%13 +-----------------+%13   a b c d e f g h"; 
		GameState gs = new GameState(text);
		String move_text = "Rd1n Re1n Hb2n Ed5n";
		ArimaaMove move = new ArimaaMove(move_text);
		LogFile.writeln(""+gs);
		LogFile.writeln(""+move_text);
		LogFile.writeln(""+move);
		gs.playMove(move);
		LogFile.writeln(""+gs);		
		move_text = "ee4n De3n ee5w De4n";
		move = new ArimaaMove(move_text);
		LogFile.writeln(""+move_text);
		LogFile.writeln(""+move);
		gs.playMove(move);
		LogFile.writeln(""+gs);
		gs=new GameState();
		move_text="Ee5 Mc2 Hb4 Hg3 Dd2 De3 Cb2 Cf2 Ra1 Rb1 Rc1 Rf1 Rg1 Rh1 Ra2 Rh2";
		move=new ArimaaMove(move_text);
		gs.playMove(move);
		LogFile.writeln("No more tests");
	}
}
