package arimaa3;

// TODO move pruning ... position evaluation should have 2 dimensions ... 
//  1) average evaluation to be used for selecting best move
//  2) sharpness ... optimistic evaluation equals average evaluation + sharpness could be used for pruning
// May be evaluation + sharpness + "other move dimension" should be used instead
// other move dimension could evaluate move history consistency ... 

// TODO goal prunning ... when there is goal after a move, find the goal and bitmap of changed positions and their surrounding 
// is "zone of action" any move not changing "zone of action" is losing as well (of course except goals) ... 
// there could be several goals so several zones of action at the same time, maintaining all of them would be expensive
// this would be complicated to handle goal allowers this way ...
// May be killer goals would suffice for that ... killer goals should be part of the goal testing
// No just reorder goal squares to speed the goal tests, testing validity of killer goals would be slow compared to it
// OK reordering goal squares is good idea, but prunning moves is late ... it would be better not to generate them at all
// if there is goal threat a lot of moves neednot be generated at all ... if there are independent steps, first steps should 
// affect the zone of action, remaining steps could be wherever. ... zone0, zone1, zone2, zone3, zone4 ... what could affect the zone 
// in given number of steps. Adding trap defender when there is our piece in lonely defended trap nearby belongs to "nearby"
// Don't forget to drags which could affect "nearby". Until zone0 is affected each step should be in zoneR where R is steps remaining.
// This could reduce the number of moves generated(and prunned afterwards) by very big amount.
// search could go deeper in goal preventing branches?? Anyways determining the goal could help.

// TODO negascout rather to negamax
import ai_util.*;

import java.util.*;

/**
 * Alpha beta search based on turns
 * @author hroch
 *
 */
public class ArimaaTurnBasedEngine extends ArimaaBaseClass implements Runnable {
	
	/** just constructor
	 */
	public ArimaaTurnBasedEngine() {
	}

	/** constant to distinguish turns got from transposition table in the processing
	 */
	private static final int ORDER_HASH = 1003;
	/** constant to distinguish killer turns in the processing
	 */
	private static final int ORDER_KILLER_TURN = 1002;
	/** constant to distinguish generated captures from all generated moves
	 */
	public static final int ORDER_CAPTURE = 1001;

	/** should the bot resign when all moves were found to be losing?
	 *  ... on last move ... should it resign earlier?
	 */
	public boolean resigning=false;
	// Sets the maximum search depth (excluding extensions)
	private int max_search_depth = MAX_TURN_PLY;
	private GenTurn gen_turn = null;	
	private TestForGoal goal = null;
	private GenCaptures captures = null;
	private MoveList root_moves = new MoveList(true); 

	private static boolean use_tricks = true;
	private static boolean use_hash_table = true;
	private static boolean use_null_move = false;

	/** debugging the search
	 */
	public static boolean show_alphabeta_search_trace = false;
	private static int trace_level = 4;

	private HashTable hash_table = new HashTable(500000);
//  	private HashTable hash_table = new HashTable(10000);

	// Working variables for gen_moves
	private MoveList turn_list_stack = new MoveList(true);
	private MoveList PV_stack = new MoveList(false);
	private ArimaaMove turn_stack[] = new ArimaaMove[MAX_TURN_PLY+2];
	private ArimaaMove best_turn_stack[] = new ArimaaMove[MAX_TURN_PLY+1];
	private static final int MAX_KILLERS = 10;
	private ArimaaMove killer_turn[][] = new ArimaaMove[MAX_TURN_PLY+1][MAX_KILLERS];// addressed by ply
	// Used to test if initial position restored
	// Statistics collection stuff
	private long ab_nodes_searched = 0;
	private long hash_hits = 0;
	private long hash_calls = 0;
	private long useless_cuts = 0;
	private long null_move_calls = 0;
	private long null_move_cuts = 0;

	/** allows PLY limits
	 * @param max_depth
	 */
	public void setMaxSearchDepth(int max_depth) {
		// Force condition 0 <= max_search_depth <= MAX_PLY
		this.max_search_depth = Math.min(max_depth, MAX_TURN_PLY);
		this.max_search_depth = Math.max(0, max_search_depth);
	}

	TimeControl time_control = new TimeControl();
	private ArimaaEvaluate eval = new ArimaaEvaluate();

	/**
	 * Generates a move for the arimaa server
	 * 
	 * @param info ArimaaServerInfo
	 * @return MoveInfo
	 */
	public MoveInfo genMove(ArimaaServerInfo info) {
		// Setup time control stuff
		ArimaaClockManager clock = new ArimaaClockManager();
		SearchTimes times = clock.calculate(info);
		return genMove(info, times);
	}

	/** Used for testing Generates a move using fixed search time
	 * 
	 * @param search_position
	 * @param fixed_search_time
	 * @return
	 */
	public MoveInfo genMove(GameState search_position, int fixed_search_time) {

		SearchTimes times = new SearchTimes();
		times.base_time_sec = fixed_search_time;
		times.panic_time_sec = fixed_search_time;
		times.mate_time_sec = fixed_search_time;

		ArimaaServerInfo info = new ArimaaServerInfo(search_position);

		return genMove(info, times);
	}

	/**
	 * Common genmove entry point
	 * 
	 * @param info ArimaaServerInfo
	 * @param times SearchTimes
	 * @return MoveInfo
	 */
	public MoveInfo genMove(ArimaaServerInfo info, SearchTimes times) {

		if (info.gs == null) { 
			info.gs = new GameState(info.position);
			info.gs.ChangeRules("RrCcDdHhMmEe", 5);
		}
		info.gs.steps_remaining=info.RootStepLimit;
		if (info.OnceRootStepLimit!=0) {
			info.gs.steps_remaining=info.OnceRootStepLimit;
			info.OnceRootStepLimit=0;
		}
		//LogFile.message("Position: " + info.position);
		resetStats();

		// Print some game info in the log file!
		LogFile.message("Opponent: " + info.enemy_name);
		LogFile.message((new Date()).toString());
		LogFile.message("\"" + info.gs.toEPDString() + "\",");
		// Formatted for array use
		LogFile.message(info.gs.toBoardString());
		LogFile.message(info.gs.toSetupString());
		LogFile.message("Max Search Depth: " + this.max_search_depth);

		LogFile.message("TC: " + times.base_time_sec + " "
				+ times.panic_time_sec + " " + times.mate_time_sec);
		time_control.setNominalSearchTime(times.base_time_sec * 1000);
		time_control.setPanicSearchTime(times.panic_time_sec * 1000);
		time_control.setMateSearchTime(times.mate_time_sec * 1000);
		time_control.setSearchScoreSlope(800); // Panic score threshold
		time_control.setSearchDepthThreshold(1);
		// Search will be at least this deep
		time_control.setSearchStartTime();

		if (info.gs.getTurnNumber() == 1) {
			ArimaaEngineInterface.chat("Hi "+info.enemy_name+" good luck!");
			ArimaaEngineInterface.chat("My handler is Hippo (Vladan Majerech)."); 
			ArimaaEngineInterface.chat("So far my position evaluation function is poor and buggy, you can see it from chat unless you write 'quiet' command.");
			ArimaaEngineInterface.chat("Your chat (especially evaluation suggestions) are welcomed");
			ArimaaEngineInterface.chat("So far I am chatting about evaluation only. + means gold leading, - silver leading");
			// Its the first move, call initial position generator
			FirstMove first_move = new FirstMove();
			long random_seed = System.currentTimeMillis();
			LogFile.message("First move random seed: " + random_seed);
			String text = first_move.getFirstMove(info.gs, random_seed);
			MoveInfo move = new MoveInfo();
			move.move_text = text;
			move.chat = "";
			return move;
		}

		if (info.repetition_rules.forced_move!="") {
			MoveInfo move = new MoveInfo();
			move.move_text = info.repetition_rules.forced_move;
			return move;
		}
		// Its not the first move, so we have to search
		return engine_control_code(info);
	}

	/**
	 * Starts the engine in a separate thread. It will search the given
	 * position. The caller is required to setup the time control stuff
	 * 
	 * @param position
	 *            ArimaaPosition
	 * @return MoveInfo
	 */
	private MoveInfo engine_control_code(ArimaaServerInfo info) {
		// Start the time control thread
		thread_info = info;
		Thread engine_control_thread = new Thread(this);
		engine_control_thread.start();

		// Sit here until time is expired
		engine_search_completed = false; // mate score
		thread_mi = null; // best move found
		current_search_depth = -1;

		while ((!time_control.isTimeExpired(current_best_score,
				current_search_depth) && !engine_search_completed)
				|| (thread_mi == null)) {
			try {
				Thread.sleep(100);
			} catch (Exception e) {//TODO only specific ones!!
			}
		}

		abort_search(); // Kills the engine thread call both!
		LogFile.message("Time Expired "+Util.toTimeString(time_control
				.getElapsedSearchTime()));
		// Dump out some search stats
		LogFile.message(getStats());

		// Wait for engine thread to close
		long start_time = System.currentTimeMillis();
		while (!engine_search_completed) {
			try {
				Thread.sleep(100);
			} catch (Exception e) {//TODO only specific ones!!
			}
		}
		long wait_time = System.currentTimeMillis() - start_time;
		LogFile.message("Wait Time: " + wait_time);

		return thread_mi;

	}

	// Variables used by engine thread
	ArimaaServerInfo thread_info;
	MoveInfo thread_mi;
	boolean engine_search_completed;

	/**
	 * This is where the engine control thread starts
	 */
	public void run() {
		// Start searching
		iterativeDeepeningRoot(thread_info);
	}

	private int initial_search_depth = 1;

	// This is used by time control to decide when to stop the search.
	private int current_best_score = -SCORE_MATE;
	private int current_search_depth = 0;
	private GameState position=new GameState();

	/**
	 * Another attempt at Root Move searching
	 * @param position ArimaaPosition
	 * @return MoveInfo
	 */

	private MoveInfo iterativeDeepeningRoot(ArimaaServerInfo info) {
		// These are used by time control to decide how much time to use
		position.copy(info.gs);
		current_best_score = -SCORE_MATE;
		current_search_depth = 0;

		ArimaaMove best_turn = new ArimaaMove();
		resetStats();
		enable_search();

		int offset = setup_search(info);
		LogFile.message("Score Offset: " + offset);
		
		turn_list_stack.clear();
		//LogFile.message("IDA:"+position);
		MoveList root_turns = genRootTurns(position);
		processRootTurns(position, root_turns);
		//LogFile.message("IDA (afterRootTurns processed) :"+position);
		// ^ mainly to log some statistics, but filtering was already done ...

		// Search until time is expired!
		try {
			for (int depth = initial_search_depth; depth <= max_search_depth; depth++) {
				//LogFile.message("IDA (starting depth "+depth+") :"+position);
				current_search_depth = depth;
				int alpha = -SCORE_MATE;
				int beta = SCORE_INFINITY;
				LogFile.message("prunning lost started "+Util.toTimeString(time_control.getElapsedSearchTime()));
				root_turns.levelPruneLosing();
				LogFile.message("prunning lost ended "+Util.toTimeString(time_control.getElapsedSearchTime()));
				if (root_turns.levelSize()>1) {
					LogFile.message("sort started "+Util.toTimeString(time_control.getElapsedSearchTime()));
					root_turns.levelSort();
					LogFile.message("sort ended "+Util.toTimeString(time_control.getElapsedSearchTime()));
					LogFile.writeln(root_turns.toString());
					for (root_turns.levelFirst();root_turns.levelCont();root_turns.levelNext()) {
						ArimaaMove turn = root_turns.getCurrentMove();
						if (show_alphabeta_search_trace) {
							SearchTrace(0, depth, "Root turn: " + turn + turn.move_ordering_value);
						} else if (info.sid.equals("analyze")) {
							LogFile.write(".");
						}
						int temp_score;
						if (turn.move_ordering_value >= 30000
								|| turn.move_ordering_value <= -30000) {
							temp_score = turn.move_ordering_value;
						} else {
							// Get a score for this move
							//LogFile.message("IDA playing "+turn+" from "+position);
							position.playMoveFull(turn);
							//LogFile.message("IDA played "+turn+" to "+position);
							temp_score = SearchPosition(position, depth,
									alpha, beta);
							//LogFile.message("IDA unplaying "+turn+" from "+position);
							position.unplayMoveFull(turn);
							//LogFile.message("IDA unplayed "+turn+" to "+position);
						}
						turn.move_ordering_value = temp_score; // better move should be searched earlier
						// We have a new best move
						// Save the best move and score
						if ((temp_score > alpha) || ((alpha<=SCORE_FORCED_LOSS) && (temp_score == alpha))){
							MoveInfo mi = new MoveInfo();
							mi.eval_score = temp_score;
							//LogFile.message("preOfficialNotation "+position);
							mi.move_text = gen_turn.getOfficialArimaaNotation(
									position, turn);
							//LogFile.message("postOfficialNotation "+position);
							hash_table.RecordHash(position, depth,
									alpha, HashTable.LOWER_BOUND,
									turn, false);
							//LogFile.message("PrePV "+position);
							mi.pv = getPV();
							//LogFile.message("PostPV "+position);
							mi.nodes_searched = this.ab_nodes_searched;
							mi.ply = depth;
							mi.search_time_ms = time_control.getElapsedSearchTime();
							if (temp_score > alpha) {
								thread_mi = mi;
								current_best_score = alpha = temp_score;
								best_turn.copy(turn);
								// This is a *LOWER* bound on the score!
								// There could be more moves to search!!!
								// Record in hash so PV can be figured out
								// report the results of the move
							}							
							// Display search status
							String text = "D:" + Util.Pad(2, depth);
							text += " "
								+ Util.toTimeString(time_control.getElapsedSearchTime());
							text += " "
								+ Util.LeftJustify(8,convertScore(alpha));
							text += "   [" + mi.move_text+"] "+ mi.pv;
							LogFile.message(text);

							// If we have a winning mate score we are done!
							if (alpha >= SCORE_FORCED_WIN) {
								break;
							}
						}
					}
				} else if (root_turns.levelSize()==1) {
					root_turns.levelFirst();best_turn.copy(root_turns.getCurrentMove());
					MoveInfo mi = new MoveInfo();
					mi.eval_score = current_best_score;
					mi.move_text = gen_turn.getOfficialArimaaNotation(
							position, best_turn);
					mi.pv = getPV();
					mi.nodes_searched = this.ab_nodes_searched;
					mi.ply = depth;
					mi.search_time_ms = time_control.getElapsedSearchTime();
					thread_mi = mi;
					break;
				} else if (root_turns.levelSize()==0) {
					if (depth==1) {
						MoveInfo mi = new MoveInfo();
						mi.eval_score = current_best_score;
						if (resigning) {
							mi.move_text = "resign";
						} else {// there must be stay a move on the place even when it is considered lost
							root_turns.levelFirst();best_turn.copy(root_turns.getCurrentMove());
							mi.move_text = gen_turn.getOfficialArimaaNotation(
									position, best_turn);
						}
						mi.pv = getPV();
						mi.nodes_searched = this.ab_nodes_searched;
						mi.ply = depth;
						mi.search_time_ms = time_control.getElapsedSearchTime();
						thread_mi = mi;
					}
					break;
				}
				// Root move loop
				// Prepare iteration status report
				// Display search status
				String text = "F:" + Util.Pad(2, depth);
				text += " "
						+ Util.toTimeString(time_control.getElapsedSearchTime());
				text += " "
						+ Util.LeftJustify(7, convertScore(current_best_score));
				text += "    ["+thread_mi.move_text;
				text += "] Nodes: " + ab_nodes_searched;
				text += " kNPS: " + ab_nodes_searched
						/ (time_control.getElapsedSearchTime() + 1);

				LogFile.message(text);
				//if (thread_info.talking) {
				//	ArimaaEngineInterface.chat("Best on depth "+depth+":"+thread_mi.move_text);
				//}
				// If we have a mate score we are done!
				if (isMateScore(alpha)) {
					break;
				}
			}
		} catch (AbortSearchException ex) {
			LogFile.message("Search Aborted! "+System.currentTimeMillis());
		}
		//LogFile.message("IDA (loop ended) :"+position);
		//if (thread_mi.eval_score<=SCORE_FORCED_LOSS) {// any hope extension
		//		position.copy(info.gs);
		//		if (root_turns.levelCont()) {
		//		ArimaaMove turn = root_turns.getCurrentMove();
		//		thread_mi.move_text = gen_turn.getOfficialArimaaNotation(
		//				position, turn);
		//		thread_mi.pv = getPV();// does not agree with played turn
		//		thread_mi.nodes_searched = this.ab_nodes_searched;
		//		thread_mi.search_time_ms = time_control.getElapsedSearchTime();
		//	}
		//}
		if (info.talking) {
			position.copy(info.gs);
			ArimaaMove turn = new ArimaaMove(thread_mi.move_text);
			//LogFile.message("IDA talk... playing "+turn+" from "+position);
			position.playMoveFull(turn);
			//LogFile.message("IDA talk... played "+turn+" to "+position);
			int score = eval.setEvalText(position);
			thread_mi.chat = "post move score "+score+" "+eval.getEvalText();
		}

		//LogFile.message("Final prunning lost started"+Util.toTimeString(time_control.getElapsedSearchTime()));
		//root_turns.levelPruneLosing();
		//LogFile.message("Final prunning lost ended"+Util.toTimeString(time_control.getElapsedSearchTime()));
		//LogFile.writeln(root_turns.toString());
		engine_search_completed = true;
		return thread_mi;
	}

	// Returns true iff score is a forced mate
	public static boolean isMateScore(int score) {
		if (Math.abs(score) > SCORE_FORCED_WIN) {
			return true;
		}
		return false;
	}

	/**
	 * Converts score to a text string
	 * 
	 * @param score int
	 * @return String
	 */
	private String convertScore(int score) {
		String result = score + "";
		if (score >= arimaa3.Constants.SCORE_FORCED_WIN) {
			result = "WON " + (SCORE_MATE - score + 1);
		}
		if (score <= arimaa3.Constants.SCORE_FORCED_LOSS) {
			result = "LOST " + (SCORE_MATE + score + 1);
		}
		return result;
	}

	/**
	 * Generate all possible turns from given position
	 * 
	 * @param initial
	 * GameState
	 * @return MoveList
	 */

	public MoveList genRootTurns(GameState root_position) {
		// is it the upperbound for number of moves?
		//LogFile.writeln("genRootTurns" + root_position.toString());
		root_moves.clear();
		gen_turn.genAllTurnsEvaluated(root_position, root_moves, eval);
		return root_moves;
	}

	public void processRootTurns(GameState position, MoveList root_moves) {

		// Setup stats
		int opponent_goals = 0;
		int opponent_captures = 0;
		int normal_moves = 0;
		int game_over = 0;
		int repetition_banned = 0;

		// Process moves
		for (root_moves.levelFirst();root_moves.levelCont();root_moves.levelNext()) {
			ArimaaMove turn = root_moves.getCurrentMove();

			//LogFile.message("processRootTurns playing "+turn+" from "+position);
			position.playMoveFull(turn);
			//LogFile.message("processRootTurns played "+turn+" to "+position);
			// Check if game is over
			if (position.isGameOver()) {
				turn.move_ordering_value = position.getGameResult();
				game_over++;
			}

			// Check for repetition banned positions
			else if (isRepetitionBanned(position.getPositionHash())) {
				turn.move_ordering_value = -SCORE_MATE;
				repetition_banned++;
			}

			// Check if opponent can goal
			else if (goal.test(position)) {
				turn.move_ordering_value = -SCORE_MATE + 1;
				opponent_goals++;
			}

			// else if (captures.testCaptures_bb(gs, FULL_BB)) {
			//	move.move_ordering_value = -ORDER_CAPTURE;
			//	opponent_captures++;
			//}
			// Must be a normal move
			else {
				// turn.move_ordering_value = -eval.Evaluate(gs)
				// already done by genAllTurnsEvaluated (including pass_step evaluation) 
				normal_moves++;
			}
			// System.out.println(move+" "+move.move_ordering_value);
			//LogFile.message("processRootTurns unplaying "+turn+" from "+position);
			position.unplayMove(turn);
			//LogFile.message("processRootTurns unplayed "+turn+" to "+position);
		}
		position.compute_secondary_bitboards();

		// Report stats ... actually the immediately lost turns are filtered already in genAllTurns
		LogFile.message("Root Moves: " + root_moves.levelSize());
		LogFile.message("Game Over: " + game_over);
		LogFile.message("Repetition Banned: " + repetition_banned);
		LogFile.message("Opponent Goals: " + opponent_goals);
		LogFile.message("Opponent Captures: " + opponent_captures);
		LogFile.message("Normal Moves: " + normal_moves);

	}

	/**
	 * Checks if given hash code is a repetition banned position
	 * 
	 * @param hash_code long
	 * @return boolean
	 */
	private boolean isRepetitionBanned(long hash_code) {
		if (thread_info.repetition_rules.positionMoves.containsKey(hash_code)) {
			if (thread_info.repetition_rules.follow_repetition[thread_info.side_to_move]) {
				return true;
			} else if (thread_info.repetition_rules.positionForbidden.contains(hash_code)) {
				return true;
			}
		}
		return false;
	}

	public void resetStats() {
		ab_nodes_searched = 0;
		// q_nodes_searched = 0; why not reset?
		hash_hits = 0;
		hash_calls = 0;
		useless_cuts = 0;
		null_move_calls = 0;
		null_move_cuts = 0;
	}

	// Dumps out statistics
	public String getStats() {
		String result = "";
		result += "AB Nodes: " + ab_nodes_searched + "\n";
		result += Util.ProbStats("Hash Table:", hash_calls, hash_hits);
		result += Util.ProbStats("Null Move:", null_move_calls, null_move_cuts);
		result += "Useless Cuts: " + useless_cuts + "\n";
		return result;
	}

	public boolean can_player_goal(GameState position) {// for TestForGoal "unit tests"
		return gen_turn.checkWinningTurn(position);//goal preventTurns would be better!!!
	}

	private boolean abort_search_flag;

	/**
	 * Timer thread calls this function to abort the search
	 */
	public void abort_search() {
		abort_search_flag = true;
	}

	public void enable_search() {
		abort_search_flag = false;
	}

	
	public int setup_search(ArimaaServerInfo info) {
		int score_offset = eval.PreProcessRootPosition(info);
		return score_offset;
	}


	public void SearchTrace(int ply, int depth, int alpha, int beta,
			ArimaaMove move, int stage) {
		String result = "";

		// Output the move
		if (move.move_ordering_value == ORDER_HASH) {
			result += "HM";
		} else if (move.move_ordering_value == ORDER_KILLER_TURN) {
			result += "KT";
		} else if (move.move_ordering_value == ORDER_CAPTURE) {
			result += "CP";
		}

		result += move.toString() + " ";
		result += "[" + alpha + " " + beta + "] ";
		result += depth + " " + "(" + stage + ")";

		SearchTrace(ply, depth, result);
	}

	// Outputs the search trace
	protected void SearchTrace(int ply, int depth, String text) {
		if (ply < trace_level) {
			String result = Util.LeftJustify(3, ply + "") + "/" + Util.LeftJustify(3, depth + "");
			result+=root_moves.getCurrentMove();
			result+=turn_list_stack.getStackSequence();
			// Provide indentation
			for (int i = 0; i < ply; i++) {
				result += "  ";
			}

			result += text;

			// Debugging only
			LogFile.message(result);

			//System.out.println(result);
			//System.out.flush();
		}
	}

	/**
	 * Score for the player who played
	 * @param srch_pos
	 * @param depth
	 * @param alpha
	 * @param beta
	 * @return
	 * @throws AbortSearchException
	 */
	private int SearchPosition(GameState srch_pos, int depth, int alpha,
			int beta) throws AbortSearchException {

		int score = AlphaBeta(depth, alpha, beta, srch_pos, false, true);

		// If we have an interesting move, run verification search, not OK to null in it
		// It is faster thanks to PV, but I don't think it is necessary now without search tricks
		// May be we have to use null windows pre search?
		if (score > alpha) {
			//LogFile.write("Pre Verification Search score:"+score+" on "+position+" PV "+getPV());
			score = AlphaBeta(depth, alpha, beta, srch_pos, true, false);
			//LogFile.write("Post verification Search score:"+score+" on "+position+" PV "+getPV());
		}
		return score;
	}

	/** Score for the player who played the last move
	 * returns score minimized for the enemy
	 * Adjust alpha,beta and result to reflect proper mate bounds
	 * This localizes all mate bounds adjustments to here only
	 * Select proper ab search mode (ie first step, later step
	 * Handles negascout adjustments as well
	 * Handles saving initial position
	 * @param depth
	 * @param alpha
	 * @param beta
	 * @param ab_pos
	 * @param is_verification_search
	 * @param ok_to_null
	 * @return
	 * @throws AbortSearchException
	 */ 

	int AlphaBeta(int depth, int alpha, int beta, GameState ab_pos,
			boolean is_verification_search, boolean ok_to_null) throws AbortSearchException {
		ab_nodes_searched++;

		// Needed to handle nega scout mate scores adjustments
		if (alpha >= SCORE_MATE - 1) {
			return SCORE_MATE - 1;
		}
		if (beta <= -SCORE_MATE + 1) {
			return -SCORE_MATE + 1;
		}

		int new_alpha = (alpha > SCORE_FORCED_WIN) ? alpha + 1
			: ((alpha < SCORE_FORCED_LOSS) ? alpha - 1 : alpha);
		int new_beta = (beta > SCORE_FORCED_WIN) ? beta + 1
			: ((beta < SCORE_FORCED_LOSS) ? beta - 1 : beta);

		assert (new_alpha <= SCORE_MATE - 1);

		// Save the new initial position
		// This is required so steps can't return to the board position at
		// turn start
		turn_list_stack.levelUp();
		int result = -negaMax(depth, -new_beta, -new_alpha, ab_pos, is_verification_search, ok_to_null);
		turn_list_stack.levelDn();
		
		int new_result = (result > SCORE_FORCED_WIN) ? result - 1
			: ((result < SCORE_FORCED_LOSS) ? result + 1 : result);
		return new_result;
	}

	/** Score for the player on the move
	 * returns score maximized for the player 
	 * @param depth
	 * @param alpha
	 * @param beta
	 * @param position
	 * @param is_defend_threat_search
	 * @param is_verification_search
	 * @param ok_to_null
	 * @return
	 * @throws AbortSearchException
	 */
	private int negaMax(int depth, int alpha, int beta,
			GameState position, boolean is_verification_search, boolean ok_to_null) throws AbortSearchException {

		//TODO in verification search ... is null window OK?
		// null window?!
		int best_score = -SCORE_MATE;
		int score;
		boolean is_defend_threat_search=false; // to be compatible in hash_table using
		
		int ply = turn_list_stack.stack_ind-1;
		ArimaaMove replyTurn=turn_stack[ply];
		ArimaaMove best_turn=best_turn_stack[ply];
		
		// Check for abort search
		if (abort_search_flag) {
			LogFile.message("main aborted!");
			throw new AbortSearchException();
		}
		// Check the hash table
		if (use_hash_table) {
			hash_calls++;
			int result = hash_table.ProbeHash(position.getPositionHash(), depth,
					alpha, beta, is_defend_threat_search);

			if (result != HashTable.NO_SCORE) {
				if (show_alphabeta_search_trace) {
					replyTurn.copy(hash_table.ProbeHashMove(position.getPositionHash()));
					SearchTrace(ply, depth, "Hash Hit: " + replyTurn + result);
				}

				hash_hits++;
				return result;
			}
		}

		// Check if its time to call eval
		if (depth <= 1 || ply >= MAX_TURN_PLY) {
			for(int i=0;i<killer_turn[ply].length;i++) {
				if (killer_turn[ply][i].steps>0) {// already set
					replyTurn.copy(killer_turn[ply][i]);
					if (position.isKillerConsistent(replyTurn)) {
						//LogFile.message("negaMax Killer playing "+replyTurn+" from "+position);
						position.playMoveFull(replyTurn);
						//LogFile.message("negaMax Killer played "+replyTurn+" to "+position);
						if (position.trapsOK()) {
							score=-eval.Evaluate(position);// hoping this is much faster than isTurnLegal
							//LogFile.message("negaMax traps OK unplaying "+replyTurn+" from "+position);
							position.unplayMoveFull(replyTurn);
							//LogFile.message("negaMax traps OK unplayed "+replyTurn+" to "+position);
							if ((score>alpha)||(score>best_score)) {
								if (gen_turn.isTurnLegal(position,replyTurn)) {
									//LogFile.message("negaMax post legality test playing "+replyTurn+" from "+position);
									position.playMoveFull(replyTurn);
									//LogFile.message("negaMax post legality test played "+replyTurn+" to "+position);									
									if (goal.isGoalThreat(position)) {
										score = goal_quiesce(position,replyTurn,score,beta,"killers");
									}
									if (score>best_score) {
										best_score=score;
										best_turn.copy(replyTurn);
									}
									if (score>alpha) {
										hash_table.RecordHash(position, depth, score,
												HashTable.LOWER_BOUND, replyTurn, is_defend_threat_search);
										alpha=score;
										if (score>=beta) {
											if (show_alphabeta_search_trace && ply <= trace_level) {
												SearchTrace(ply, depth, "Killer cut " + replyTurn + " "+ score + " beta " + beta);
											}
											ArimaaMove swap=killer_turn[ply][i];
											for(;i>0;i--) {
												killer_turn[ply][i]=killer_turn[ply][i-1];
											}
											killer_turn[ply][0] = swap;
											//LogFile.message("negaMax Killer beta cut unplaying "+replyTurn+" from "+position);
											position.unplayMoveFull(replyTurn);
											//LogFile.message("negaMax Killer beta cut unplayed "+replyTurn+" to "+position);											
											return score;									
										}
									}
									//LogFile.message("negaMax Killer unplaying "+replyTurn+" from "+position);
									position.unplayMoveFull(replyTurn);
									//LogFile.message("negaMax Killer unplayed "+replyTurn+" to "+position);											
								} else if (show_alphabeta_search_trace && ply <= trace_level) {
									SearchTrace(ply, depth, "Killer illegal " + replyTurn + " " + score);
								}
							}
						} else {
							//LogFile.message("negaMax Killer bad traps unplaying "+replyTurn+" from "+position);
							position.unplayMoveFull(replyTurn);
							//LogFile.message("negaMax Killer bad traps unplayed "+replyTurn+" to "+position);											
						}
					} else if (show_alphabeta_search_trace && ply <= trace_level) {
						SearchTrace(ply, depth, "Killer totally not applicable " + replyTurn);
					}
				}
			}
			//gs.compute_secondary_bitboards(); to test consistency
			captures.genCaptures(position, turn_list_stack, true);
			// it could use less than 4 steps so the remaining steps could improve it a lot, but
			// the cut probbaility is high
			for (turn_list_stack.levelFirst();turn_list_stack.levelCont();turn_list_stack.levelNext()) {
				replyTurn.copy(turn_list_stack.getCurrentMove());
				replyTurn.steps=position.steps_remaining; // play pass if needed (full evaluation needed)
				//LogFile.message("negaMax captures playing "+replyTurn+" from "+position);
				position.playMoveFull(replyTurn);
				//LogFile.message("negaMax captures played "+replyTurn+" to "+position);											
				if (goal.test(position)) {
					//LogFile.message("negaMax captures goal allower unplaying "+replyTurn+" from "+position);
					position.unplayMoveFull(replyTurn);
					//LogFile.message("negaMax captures goal allower unplayed "+replyTurn+" to "+position);											
					continue;
				}
				if (goal.isGoalThreat(position)) {
					score = goal_quiesce(position,replyTurn,-eval.Evaluate(position),beta,"captures");
				} else {
					score = -eval.Evaluate(position);
				}
				//LogFile.message("negaMax captures unplaying "+replyTurn+" from "+position);
				position.unplayMoveFull(replyTurn);
				//LogFile.message("negaMax captures unplayed "+replyTurn+" to "+position);											
				if (score>best_score) {
					best_score=score;
					best_turn.copy(replyTurn);
				}
				if (score>alpha) {
					hash_table.RecordHash(position, depth, score,
							HashTable.LOWER_BOUND, replyTurn, is_defend_threat_search);
					alpha=score;
					if (score>=beta) {
						if (show_alphabeta_search_trace && ply <= trace_level) {
							SearchTrace(ply, depth, "Capture cut " + replyTurn + " "+ score + " beta " + beta);
						}
						UpdateKillerTurns(ply, replyTurn, "1 captures cut");
						return score;
					}
				}
			}
			turn_list_stack.levelClear();
			int bscore = gen_turn.getBestEvaluatedTurnAndThreats(position, turn_list_stack, replyTurn, eval, thread_info.repetition_rules.positionForbidden, beta);
			if (bscore>=beta) {// quiesce ?!?
				UpdateKillerTurns(ply, replyTurn,"1 best cut");
				if (show_alphabeta_search_trace && ply <= trace_level) {
					SearchTrace(ply, depth, "Goal threat cut " + replyTurn + " "+ bscore + " beta " + beta);
				}
				return bscore;
			}
			for (turn_list_stack.levelFirst();turn_list_stack.levelCont();turn_list_stack.levelNext()) {
				ArimaaMove reTurn=turn_list_stack.getCurrentMove();
				//LogFile.message("negaMax goal threats playing "+reTurn+" from "+position);
				position.playMoveFull(reTurn);
				//LogFile.message("negaMax goal threats played "+reTurn+" to "+position);											
				//goal.isGoalThreat(new_gs); could be usefull in quick reply generation
				score = goal_quiesce(position,reTurn,-eval.Evaluate(position),beta,"threats");
				//LogFile.message("negaMax goal threats unplaying "+reTurn+" from "+position);
				position.unplayMoveFull(reTurn);
				//LogFile.message("negaMax goal threats unplayed "+reTurn+" to "+position);											
				if (score>best_score) {
					best_score=score;
					best_turn.copy(reTurn);
				}
				if (score>alpha) {
					hash_table.RecordHash(position, depth, score,
							HashTable.LOWER_BOUND, reTurn, is_defend_threat_search);
					alpha=score;
					if (score>=beta) {
						UpdateKillerTurns(ply, reTurn,"1 threats cut");
						if (show_alphabeta_search_trace && ply <= trace_level) {
							SearchTrace(ply, depth, "Goal threat cut " + reTurn + " "+ score + " beta " + beta);
						}
						return score;
					}
				}
			}
			//LogFile.message("negaMax best playing "+replyTurn+" from "+position);
			position.playMoveFull(replyTurn);
			//LogFile.message("negaMax best played "+replyTurn+" to "+position);											
			score = bscore;
			if (goal.isGoalThreat(position)) {
				// already processed
			} else {
				if (score>best_score) {
					best_score=score;
					best_turn.copy(replyTurn);
				}
				if (score>alpha) {
					alpha=score;
					if (score<beta) {
						hash_table.RecordHash(position, depth, score,
								HashTable.EXACT_BOUND, replyTurn, is_defend_threat_search);
					} else {
						hash_table.RecordHash(position, depth, score,
								HashTable.LOWER_BOUND, replyTurn, is_defend_threat_search);
						if (show_alphabeta_search_trace && ply <= trace_level) {
							SearchTrace(ply, depth, "Best cut " + replyTurn + " "+ score + " beta " + beta);
						}
						UpdateKillerTurns(ply, replyTurn, "1 best cut");
						//LogFile.message("negaMax best cut unplaying "+replyTurn+" from "+position);
						position.unplayMoveFull(replyTurn);
						//LogFile.message("negaMax best cut unplayed "+replyTurn+" to "+position);											
						return score;
					}
				}
			}
			//LogFile.message("negaMax best unplaying "+replyTurn+" from "+position);
			position.unplayMoveFull(replyTurn);
			//LogFile.message("negaMax best unplayed "+replyTurn+" to "+position);											
			if (show_alphabeta_search_trace && ply <= trace_level) {
				SearchTrace(ply, depth, "Best reply " + best_turn + " "+ score + " alpha " + alpha + " beta " + beta);
			}
			if (beta>=SCORE_INFINITY) {
				if (best_turn.move_ordering_value<ORDER_KILLER_TURN) {
					UpdateKillerTurns(ply, best_turn,"1 best");
				}
			}
			return best_score;
		}

		//deeper search
		// Try null move pruning
		if (use_null_move && ok_to_null) {
			int new_depth = depth - 1; // null move depth reduction?
			position.playPASSturn();
			if (show_alphabeta_search_trace && ply <= trace_level) {
				SearchTrace(ply, depth, "Trying Null move " + beta);
			}
			null_move_calls++;
			score = AlphaBeta(new_depth, beta - 1, beta,
					position, is_verification_search, false);
			position.unplayPASS();
			if (show_alphabeta_search_trace && ply <= trace_level) {
				SearchTrace(ply, depth, "Finished Null move " + score);
			}

			if (score >= beta) {
				null_move_cuts++;
				return score;
			}
		}

		// Generate all moves
		int moves_searched = 0;
		for (int stage=0;stage<=2;stage++) {
			if ((stage == 2) && (beta >= SCORE_INFINITY)) {
				stage += 1;// sort exception for the first searched move
			}
			gen_turns(ply, position, stage);
			if (turn_list_stack.levelSize()>0) {
				if (stage==3) {
					turn_list_stack.levelSort();
				}
				// generates hash move, killer moves, capture moves and normal moves
				// stages to allow beta cuts before all moves were generated (especially the killer moves)
				// for speeding up some cases stage 2 could be omitted
				// Iterate thru the moves
				for (turn_list_stack.levelFirst();turn_list_stack.levelCont();turn_list_stack.levelNext()) {
					ArimaaMove turn = turn_list_stack.getCurrentMove();

					if (turn.move_ordering_value==ORDER_KILLER_TURN) {
						if (!gen_turn.isTurnLegal(position,turn)) {
							continue;
						}
					}
					//LogFile.message("negaMax deeper playing "+turn+" from "+position);
					position.playMoveFull(turn);
					//LogFile.message("negaMax deeper played "+turn+" to "+position);											
					if (stage<2) {// stage 2 already filtered
						if (goal.test(position)) {
							//LogFile.message("negaMax deeper goal unplaying "+turn+" from "+position);
							position.unplayMoveFull(turn);
							//LogFile.message("negaMax deeper goal unplayed "+turn+" to "+position);											
							continue;
						}
					}

					if (show_alphabeta_search_trace) {
						SearchTrace(ply, depth, alpha, beta, turn, stage);
					}

					// Check if game is over, can be WIN/DRAW/LOSS
					if (position.isGameOver()) {
						// Note must search rest of possible moves!!!
						score = position.getGameResult();
					} else {
						// Game is NOT over, so we search
						int new_depth = depth - 1;

						// This routine handles mate bounds, timeouts, negamax
						score = AlphaBeta(new_depth, alpha, beta, position, is_verification_search, ok_to_null);

						// Determine if verification search is required
						// Any interesting move with depth>=1 must be looked at again
						// The verification search checks for any mate scores
						if (!is_verification_search && new_depth >= 1 && score > alpha) {
							score = AlphaBeta(new_depth, alpha, beta, position, true, ok_to_null);
						}
					}
					//LogFile.message("negaMax deeper unplaying "+turn+" from "+position);
					position.unplayMoveFull(turn);
					//LogFile.message("negaMax deeper unplayed "+turn+" to "+position);											
					if (score>best_score) {
						best_score=score;
						best_turn.copy(turn);
					}
					// Check if we got an interesting score
					if (score > alpha) {
						hash_table.RecordHash(position, depth, score,
								HashTable.LOWER_BOUND, turn, is_defend_threat_search);
						alpha = score;
						if (score >= beta) {
							if (show_alphabeta_search_trace && ply <= trace_level) {
								SearchTrace(ply, depth, "Deep cut " + turn + " "+ score + " beta " + beta);
							}
							if (turn.move_ordering_value<ORDER_KILLER_TURN) {
								UpdateKillerTurns(ply, turn,"deep stage "+stage+" cut");
							} else {
								MoveFrontKillerTurn(ply,turn,"deep stage "+stage+" cut");
							}
							return score;
						}
					}

					// Loop post operations
					moves_searched++;

					// If we get a mate score we are done
					if (alpha >= SCORE_FORCED_WIN) {
						break;
					}
				}
			}
		}
		// If we can't move, we lose!
		// Warning: This is NOT TRUE IF PRUNING IS BEING DONE!!!!
		if (moves_searched == 0) {
			if (show_alphabeta_search_trace) {
				SearchTrace(ply, depth, "No Moves!");
			}
			return -SCORE_MATE - 1;
		}

		hash_table.RecordHash(position, depth, best_score,
				HashTable.UPPER_BOUND, best_turn, is_defend_threat_search);
		if (beta>=SCORE_INFINITY) {
			if (best_turn.move_ordering_value<ORDER_KILLER_TURN) {
				UpdateKillerTurns(ply, best_turn,"deep best ");
			} else {
				MoveFrontKillerTurn(ply, best_turn,"deep best ");
			}
		}
		return best_score;

	}

	private int goal_quiesce(GameState position,ArimaaMove reTurn, int score, int beta, String msg)
		throws AbortSearchException  {//TODO
		//dummy version this should be alpha_beta with all turns generated on first player, and only goal threats on reply
		//result has to be  maximum of scores of "attacking player"?!?
		turn_list_stack.levelUp();
		int ply=turn_list_stack.stack_ind-1;
		int best_re = SCORE_MATE-2;// from the perspective of player who already played so minimizing replies
		ArimaaMove replyTurn = turn_stack[ply];
		if (abort_search_flag) {
			LogFile.message("goal_quiesce aborted!");
			throw new AbortSearchException();
		}
		if (gen_turn.genFirstTurns(position, turn_list_stack,5)) {// goal prevent turns would be better
			best_re=score;
			//int dist = turn_list_stack.levelSize()/3;
			//int offs = random.nextInt(dist);
			//for (turn_list_stack.levelFirst();turn_list_stack.levelCont();turn_list_stack.levelNext()) {
			//	if (offs==0) {
			//		offs=dist;
			//		ArimaaMove turn = turn_list_stack.getCurrentMove();
			//		new_gs.playFull(turn, gs);
			//		int reScore = gen_turn.getBestEvaluatedTurn(gs, replyTurn, eval, thread_info.repetition_rules.positionForbidden, beta);
			//		if (reScore<best_re) {
			//			best_re=reScore;
			//		}
			//	}
			//	offs--;
			//}
			//if (best_re > score+20) {
			//	score += 5;
			//}
		} else {
			for (turn_list_stack.levelFirst();turn_list_stack.levelCont();turn_list_stack.levelNext()) {
				ArimaaMove turn = turn_list_stack.getCurrentMove();
				position.playMoveFull(turn);
				int reScore = -gen_turn.getBestEvaluatedTurn(position, replyTurn, eval, thread_info.repetition_rules.positionForbidden, beta);
				position.unplayMoveFull(turn);
				if (show_alphabeta_search_trace && ply <= trace_level) {
					SearchTrace(ply, 1 , "Goal_quisce " + reTurn + turn + reScore +" "+ beta+" "+msg);
				}
				if (reScore<best_re) {
					best_re=reScore;
				}
			}
		}
		turn_list_stack.levelDn();
		if (show_alphabeta_search_trace && ply <= trace_level) {
			SearchTrace(ply, 0 , "Goal_quisce score "+reTurn+score+" deeper "+best_re +" beta " + beta+" "+msg);
		}
		if (best_re > score) {
			score = best_re;
		}
		return score;
	}
	
	private void gen_turns(int ply, GameState position, int stage) {

		// Get hash move
		turn_list_stack.levelClear();
		if (stage==0) {
			ArimaaMove turn = hash_table.ProbeHashMove(position);
			if (turn != null) {
				ArimaaMove temp = turn_list_stack.getMove();
				temp.copy(turn);
				temp.move_ordering_value = ORDER_HASH;
			}
			for(int i=0;i<killer_turn[ply].length;i++) {
				if (killer_turn[ply][i].steps>0) {// already set
					ArimaaMove temp = turn_list_stack.getMove();
					temp.copy(killer_turn[ply][i]);
				}
			}
		} else if (stage==1) {
			// Get captures
			captures.genCaptures(position, turn_list_stack, true);
		} else if (stage==2){
			// Get slide/drag moves
			gen_turn.genNonLosingTurns(position, turn_list_stack);
		} else if (stage==3) {
			gen_turn.genAllTurnsEvaluated(position, turn_list_stack,eval);
		}
	}

	public void UpdateKillerTurns(int ply, ArimaaMove betaCutTurn, String txt) {
		int last=killer_turn[ply].length-1;
		if (show_alphabeta_search_trace) {
			StringBuilder msg=new StringBuilder();
			msg.append("preU "+txt+" killers["+ply+"]:");
			for(int i=0;i<=last;i++) {
				if (killer_turn[ply][i].steps==0) {
					break;
				}
				msg.append(" "+killer_turn[ply][i]);
			}
			LogFile.writeln(msg.toString());
		}
		if (last>=0) {
			int i=last;
			if (killer_turn[ply][last].steps==0) {// not set yet
				for(;i<last;i++) {
					if (killer_turn[ply][i].steps==0) {
						break;
					}
				}
			}
			ArimaaMove swap=killer_turn[ply][i];
			for(;i>0;i--) {
				killer_turn[ply][i]=killer_turn[ply][i-1];
			}
			killer_turn[ply][0] = swap;
			swap.copy(betaCutTurn);
			swap.move_ordering_value=ORDER_KILLER_TURN;
		}
		if (show_alphabeta_search_trace) {
			StringBuilder msg=new StringBuilder();
			msg.append("postU"+txt+" killers["+ply+"]:");
			for(int i=0;i<=last;i++) {
				if (killer_turn[ply][i].steps==0) {
					break;
				}
				msg.append(" "+killer_turn[ply][i]);
			}
			LogFile.writeln(msg.toString());
		}
	}

	public void MoveFrontKillerTurn(int ply, ArimaaMove betaCutTurn, String txt) {
		int last=killer_turn[ply].length-1;
		if (show_alphabeta_search_trace) {
			StringBuilder msg=new StringBuilder();
			msg.append("preM "+txt+" killers["+ply+"]:");
			for(int i=0;i<last;i++) {
				if (killer_turn[ply][i].steps==0) {
					break;
				}
				msg.append(" "+killer_turn[ply][i]);
			}
			LogFile.writeln(msg.toString());
		}
		if (last>=0) {
			int i=0;
			for(;i<last;i++) {
				if (betaCutTurn.equals(killer_turn[ply][i])) {
					break;
				}
			}
			ArimaaMove swap=killer_turn[ply][i];
			for(;i>0;i--) {
				killer_turn[ply][i]=killer_turn[ply][i-1];
			}
			killer_turn[ply][0] = swap;
			swap.copy(betaCutTurn);
			swap.move_ordering_value=ORDER_KILLER_TURN;
		}
		if (show_alphabeta_search_trace) {
			StringBuilder msg=new StringBuilder();
			msg.append("postM"+txt+" killers["+ply+"]:");
			for(int i=0;i<last;i++) {
				if (killer_turn[ply][i].steps==0) {
					break;
				}
				msg.append(" "+killer_turn[ply][i]);
			}
			LogFile.writeln(msg.toString());
		}
	}

	public void init(){
		gen_turn = new GenTurn();
		for (int i = 0; i <= MAX_TURN_PLY; i++) {
			turn_stack[i] = new ArimaaMove();
			best_turn_stack[i] = new ArimaaMove();
			for (int j = 0; j < killer_turn[i].length; j++) {
				killer_turn[i][j] = new ArimaaMove();
				killer_turn[i][j].clear();
			}
		}
		captures = new GenCaptures(true);
		goal = new TestForGoal();
		hash_table.init();
	}
	
	{init();}
	
	/**
	 * Returns principal variation from current position (transposition table stored path)
	 * @param position
	 * @return
	 */
	public String getPV() {// TODO rewrite not using playMove at all

		//LogFile.message("getPV "+position);
		String result = "";
		int pv_moves = 0;
		PV_stack.clear();
		
		while (true) {
			// Get hash move
			ArimaaMove hash_move = hash_table.ProbeHashMove(position);
			if (hash_move == null) {
				break;
			}

			PV_stack.getMove().copy(hash_move);
			//LogFile.message("getPV playing "+hash_move+" from "+position);
			position.playMoveFull(hash_move);
			//LogFile.message("getPV played "+hash_move+" to "+position);

			// Output the move
			result += hash_move.toString() + " ";

			// Cutoff if the move list repeats
			pv_moves++;
			if (pv_moves > 50) {
				break;
			}
		}

		ArimaaMove stack_move=null;
		while ((stack_move=PV_stack.levelPop())!=null) {
			//LogFile.message("getPV stack unplaying "+stack_move+" from "+position);
			position.unplayMove(stack_move);
			//LogFile.message("getPV stack unplayed "+stack_move+" to "+position);
		}
		position.compute_secondary_bitboards();
		return result;
	}

	private static String tests[] = {
			//"4w %13 +-----------------+%138| r r   H r r r r |%137|   e   C r   r   |%136|   d X     X     |%135|   d R M c       |%134|       R         |%133|     X     X     |%132|                 |%131|       R       R |%13 +-----------------+%13   a b c d e f g h%13",
			//"2w %13 +-----------------+%138|                 |%137|                 |%136|   R             |%135|                 |%134|                 |%133|         D   r   |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
			//"3w %13 +-----------------+%138|                 |%137|                 |%136|   R             |%135|                 |%134|                 |%133|             r   |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
			//"31b %13 +-----------------+%138|           r     |%137|           E h   |%136|     X     X     |%135|               r |%134| r H C     H   r |%133| r   X   e C   R |%132| R R M D r D R R |%131|   R   R R       |%13 +-----------------+%13   a b c d e f g h%13"
			//"13w %13 +-----------------+%138| r r             |%137| r               |%136| R E             |%135|       e         |%134|                 |%133|         h       |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
			"12w %13 +-----------------+%138| r r             |%137| r c E       e   |%136| R           M   |%135|                 |%134|                 |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
	};

	public static void main(String args[]) {

		LogFile.setMessageDisplay(true);

		for (String text : tests) {
			GameState position = new GameState(text);
			ArimaaTurnBasedEngine engine = new ArimaaTurnBasedEngine();
			engine.resetStats();

			MoveInfo info = engine.genMove(position, 60000);
			System.out.println("*" + info.move_text + "*");
		}
	}
}