package arimaa3;

import java.util.*;
import ai_util.*;

// Generates the first move for each side
public class FirstMove extends ArimaaBaseClass {

	public FirstMove() {
	}

	private Random random = null;
	private arimaa3.GameState position = null;

	// Generates the initial setup
	public String getFirstMove(arimaa3.GameState start_position,
			long random_seed) {
		String result = "";
		random = new Random(random_seed);
		this.position = start_position;

		if (position.getSideToMove() == PL_GOLD) {
			result = getFirstMoveWhite();
		}
		if (position.getSideToMove() == PL_SILVER) {
			result = getFirstMoveBlack();
		}
		return result;

	}

	private final int E = 0;
	private final int M = 1;
	private final int H1 = 2;
	private final int H2 = 3;
	private final int D = 4;
	private final int C = 5;
	private final int R = 6;

	int allowed[] = new int[7];
	int forced[] = new int[7];
	int reqCnt[] = { 1, 1, 1, 1, 2, 2, 8 };

	private void resetAllowed() {
		allowed[E] = 0x3c00;
		allowed[M] = 0x5A00;
		allowed[H1] = 0x4200;
		allowed[H2] = 0xff00;
		allowed[D] = 0xff5a;
		allowed[C] = 0xA57e;
		allowed[R] = 0xa5ff;
		for (int i = 0; i < forced.length; i++) {
			recalcForced(i);
		}
	}

	private void recalcForced(int piece) {
		forced[piece] = 0xffff;
		for (int i = 0; i < allowed.length; i++) {
			if (i != piece)
				forced[piece] &= ~allowed[i];
		}
	}

	// returns a random selection from the valid set of columns
	// or -1 if valid set is empty
	private int getOneValid(int mask) {
		if (mask == 0)
			return 0; // no match possible

		int temp = random.nextInt(Util.PopCnt(mask));
		while (temp-- > 0) {
			mask &= ~Util.LastBit(mask);
		}
		return Util.LastBit(mask);
	}

	private boolean ForcePiece(int piece,int recur_done) {
		int recur_to_do = 1 << piece;
		while (recur_to_do > 0) {
			int mask = allowed[piece];
			for (int i = 0; i < allowed.length; i++) {
				if (i != piece) {
					allowed[i] &= ~mask;
					int cnt = Util.PopCnt(allowed[i]);
					if (cnt < reqCnt[i])
						return false;
					if (cnt == reqCnt[i])
						recur_to_do |= 1 << i;
				}
			}
			recur_done |= 1 << piece;
			recur_to_do &= ~recur_done;
			piece = Util.FirstOne(recur_to_do);
		}
		return true;
	}

	private boolean SetPiece(int piece, int done_mask) {
		// if (allowed[piece]==0) return false; checked by MaskOut
		recalcForced(piece);
		if ((forced[piece] & (~allowed[piece])) != 0)
			return false;
		int forcedCnt = Util.PopCnt(forced[piece]);
		int allowedCnt = Util.PopCnt(allowed[piece]);
		if (forcedCnt > reqCnt[piece])
			return false;

		if (2 * reqCnt[piece] < forcedCnt + allowedCnt) {
			// selecting to use
			while (forcedCnt < reqCnt[piece]) {
				forcedCnt++;
				forced[piece] |= getOneValid(allowed[piece] & ~forced[piece]);
			}
			allowed[piece] = forced[piece];
			if (!ForcePiece(piece,done_mask))
				return false;
			assert (Util.PopCnt(allowed[piece]) == reqCnt[piece]);
		} else { // selecting to discard
			while (allowedCnt > reqCnt[piece]) {
				allowedCnt--;
				allowed[piece] &= ~getOneValid(allowed[piece] & ~forced[piece]);
			}
			forced[piece] = allowed[piece];
			if (!ForcePiece(piece,done_mask))
				return false;
			assert (Util.PopCnt(allowed[piece]) == reqCnt[piece]);
		}
		return true;
	}

	private String getIndexText(int index, int colour) {
		assert (colour == PL_GOLD || colour == PL_SILVER);
		assert (index >= 0 && index <= 15);

		String result = "";

		// Get the column
		result += Notation.Col2Name(index & 0x07);

		// Get the row
		if (colour == PL_GOLD) {
			result += (index <= 7) ? "1" : "2";
		}
		if (colour == PL_SILVER) {
			result += (index <= 7) ? "8" : "7";
		}
		return result;
	}

	// reports all pieces of given kind
	private String reportPiece(int piece, String piecechar, int colour) {
		String result = "";
		int posMask = forced[piece];
		for (int i = 0; i < reqCnt[piece]; i++) {
			int index = Util.FirstOne(posMask);
			posMask &= ~(1 << index);
			result += " " + piecechar + getIndexText(index, colour);
		}
		return result;
	}

	private String getFirstMoveWhite() {
		boolean fail = true;
		int done=0;
		while (fail) {
			resetAllowed();

			if (!SetPiece(E,done))
				continue; // Position the elephant

			if (!SetPiece(M,done|=(1<<E)))
				continue; // Position the Camel

			if (!SetPiece(H1,done|=(1<<M)))
				continue; // Position the first Horse

			// second horse allowed on border only when near the other
			int Hneighbour = ((forced[H1] << 1) | (forced[H1] >>> 1));
			allowed[H2] &= (~0x8100) | Hneighbour;
			if (!SetPiece(H2,done|=(1<<H1)))
				continue; // Position the second Horse

			// border horse does not like rabbit under trap
			if ((forced[H2] & 0x8100) != 0) {
				allowed[R] &= (forced[H2] == 0x8000) ? ~0x2000 : ~0x0400;
			}

			if (!SetPiece(D,done|=(1<<H2)))
				continue; // Position the dogs

			if (!SetPiece(C,done|=(1<<D)))
				continue; // Position the cats

			if (!SetPiece(R,done|=(1<<C)))
				continue; // Position the rabbits (it already is)

			fail = false;
		}
		// we are successfull now so report the position

		String result = "";
		result += reportPiece(E, "E", PL_GOLD);
		result += reportPiece(M, "M", PL_GOLD);
		result += reportPiece(H1, "H", PL_GOLD);
		result += reportPiece(H2, "H", PL_GOLD);
		result += reportPiece(D, "D", PL_GOLD);
		result += reportPiece(C, "C", PL_GOLD);
		result += reportPiece(R, "R", PL_GOLD);

		return result;
	}

	/*
	 * gets the first column of the white piece private int
	 * getWhitePieceMask(int piece) { int result = 0; for (int row = 0; row < 2;
	 * row++) { for (int col = 0; col < 8; col++) { int piece_type =
	 * position.getPieceType(Notat.RowCol2Index(row, col)); if (piece_type ==
	 * piece) { result |= 1<<((1-row)*8+col); } } } return result; }
	 */
	private String getFirstMoveBlack() {

		int e_mask = (int) position.stronger_or_eq_bb[5];
		int m_mask = (int) position.stronger_or_eq_bb[4] ^ e_mask;
		int h_mask = (int) position.stronger_or_eq_bb[3] ^ m_mask;
		int d_mask = (int) position.stronger_or_eq_bb[2] ^ h_mask;
		int done=0;
		
		boolean fail = true;
		while (fail) {
			resetAllowed();

			// Rule 1: Do NOT line up elephant with central elephant
			allowed[E] &= ~(e_mask & 0x1C00);

			// Rule 2: No horse on camel wing border
			allowed[H2] &= (((Util.FirstOne(m_mask) & 0x07) < 4) ? ~0x0100
					: ~0x8000);

			// Rule 3: If unballanced horses, dogs on opposite side
			if ((h_mask & 0x1f0f) == 0) {
				allowed[D] &= 0xf0f0;
			}
			if ((h_mask & 0xf1f0) == 0) {
				allowed[D] &= 0x0f0f;
			}

			// Rule 4: No dogs line up with horses on front
			allowed[D] &= ~(h_mask & 0xff00);

			// Rule 5: No cats line up with dogs or horses on front except
			// under traps
			allowed[C] &= ~((h_mask | d_mask) & 0xdb00);

			if (!SetPiece(E,done))
				continue; // Position the elephant
			if (!SetPiece(M,done|=(1<<E)))
				continue; // Position the Camel
			if (!SetPiece(H1,done|=(1<<M)))
				continue; // Position the first Horse

			// second horse allowed on border only when near the other or
			// camel
			int MHneighbour = ((forced[H1] | forced[M]) << 1)
					| ((forced[H1] | forced[M]) >>> 1);
			allowed[H2] &= (~0x8100) | MHneighbour;
			if (!SetPiece(H2,done|=(1<<H1)))
				continue; // Position the second Horse

			// border horse does not like rabbit under trap
			if ((forced[H2] & 0x8100) != 0) {
				allowed[R] &= (forced[H2] == 0x8000) ? ~0x2000 : ~0x0400;
			}

			if (!SetPiece(D,done|=(1<<H2)))
				continue; // Position the dogs
			if (!SetPiece(C,done|=(1<<D)))
				continue; // Position the cats
			if (!SetPiece(R,done|=(1<<C)))
				continue; // Position the rabbits (it already is)

			fail = false;
		}
		// we are successfull now so report the position

		String result = "";
		result += reportPiece(E, "e", PL_SILVER);
		result += reportPiece(M, "m", PL_SILVER);
		result += reportPiece(H1, "h", PL_SILVER);
		result += reportPiece(H2, "h", PL_SILVER);
		result += reportPiece(D, "d", PL_SILVER);
		result += reportPiece(C, "c", PL_SILVER);
		result += reportPiece(R, "r", PL_SILVER);

		return result;
	}

	public static void main(String args[]) {

		String text4[] = {
				"1w %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|                 |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
				"1b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|                 |%133|                 |%132| C D H M E H D C |%131| R R R R R R R R |%13 +-----------------+%13   a b c d e f g h%13",
				"1b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|                 |%133|                 |%132| C D H C R H D M |%131| R R R E R R R R |%13 +-----------------+%13   a b c d e f g h%13",
				"1b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|                 |%133|                 |%132| M D H C E H D C |%131| R R R R R R R R |%13 +-----------------+%13   a b c d e f g h%13",
				"1b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|                 |%133|                 |%132| C M D H E D H C |%131| R R R R R R R R |%13 +-----------------+%13   a b c d e f g h%13", };

		FirstMove first_move = new FirstMove();

		int seed = 1773;
		for (String pos : text4) {
			for (int t=0;t<5;t++){
				seed++;
				arimaa3.GameState position = new arimaa3.GameState(pos);
				System.out.println(position.toBoardString());
				String result = first_move.getFirstMove(position, seed);
				System.out.println(result);
			}
		}
	}
}