package arimaa3;

import ai_util.*;

import java.util.*;

// TODO maintain basic info static and complicated computed info dynamic
// maintain stack of computed info to allow quick return to previous state in DFS
// while undoing the basic info directly. The basic info includes at least the board, 
// but basic bitboards could be here as well.
// be carefull to maintain basic info correctly when processing exceptions!
// TODO stronger_or_eq_bb without player dimension and player_bb[2] determining colors ??
// piece_at again without the color dimension

/**
 * Class represents the board and generates steps in the position. it does not generate full turns
 * the board is represented by bitboards by the pieces strength ... the basis is stronger_or_eq[strength],
 * but for faster evaluation board of 64 positions piece_at[] is maintained as well.
 * This causes problems with copying huge representation of position and updating the piece_at 
 * with otherwise undirected moves is problem. This is why whole engine was rewritten to use do/undo with 
 * just one maintained position rather to copying positions. 
 * @author hroch
 *
 */
public class GameState extends ArimaaBaseClass {
	
	/** constructor
	 * clears the created board
	 */
	public GameState() {
		Init();
	}

	/** are we debugging?
	 * 
	 */
	public boolean testing = true;
	private boolean inconsistent = false;
	
	/**
	 * An attempt to speedup computation by reducing the number of bitboards when some 
	 * strength does not appear more on the board. Done between moves in idle time.
	 * In typical Arimaa game this happens once or twice per game. And question is how big speedup it is.
	 * It tries to maintain Zobrist hashes unchanged by the transformation.
	 * @param new_InclPieceNames ... Symbolic names of pieces remaining on the board
	 * @param new_Strongest ... the number of different strengths minus 1
	 * @return true if shrinked
	 */
	public boolean ChangeRules(String new_InclPieceNames, int new_Strongest) {
		// this routine could be much simpler if we don't want Zobrist hashing to be 
		// compatible among several games. Actually it would bring no match in transposition table
		// so it is mainly just an exercise
		if ((new_InclPieceNames.equals(InclPieceNames))&&(Strongest == new_Strongest)) {
			// no rules changed
			return false;	
		}
		LogFile.message("Changing rules from ("+InclPieceNames+","+Strongest+")"+this);
		checkZobr(); // ensure there was consistency
		long new_Zobr[][][] = new long[6][][];
		long new_stronger_or_equal_bb[]=new long[new_Strongest+2];
		new_stronger_or_equal_bb[new_Strongest+1]=0;
		for(int i=5;i>=0;i--) {
			String pieces = new_InclPieceNames.substring(i<<1,(i+1)<<1);
			int oriPos = InclPieceNames.indexOf(pieces)>>1;
			new_Zobr[i]=Zobr[oriPos];
			if (i<=new_Strongest) {
				if (oriPos<=Strongest) {
					new_stronger_or_equal_bb[i]=stronger_or_eq_bb[oriPos];
				} else {
					// case of expansion (new game)
					new_stronger_or_equal_bb[i]=new_stronger_or_equal_bb[i+1];
				}
			}
		}
		for(int i=0;i<64;i++) {
			if (piece_at[i]>=0) {
				piece_at[i]=new_InclPieceNames.indexOf(InclPiece2Name(piece_at[i]));
			}
		}
		InclPieceNames = new_InclPieceNames;
		Strongest = new_Strongest;
		stronger_or_eq_bb = new_stronger_or_equal_bb;
		Zobr = new_Zobr;
		//Material.debug=true;
		mat  = Material.getMaterial(this);
		//Material.debug=false;
		//checkZobr(); // ensure consistency remains
		LogFile.message("Changed rules to ("+InclPieceNames+","+Strongest+")"+this);
		return true;
	}
	
	/**
	 * @return true if and only if shrinked
	 */
	
	/** Looks at the material used in the position and changes rules to as small as possible
	 * @return true if shrinked
	 */
	public boolean Compress() {
		if (turn<=1) return false;
		// to prevent shrink on empty board
		int strength_mask=0;
		//strength_mask |= 1 << strength;
		String ori_unusedPieceNames = InclPieceNames.substring((Strongest+1)<<1);
		String usedPieceNames = "", unusedPieceNames = "";
		long stronger_bb = 0, current_bb = 0;
		for(int s=Strongest;s>=0;s--) {
			current_bb = (stronger_or_eq_bb[s])^stronger_bb;
			if (current_bb!=0) {
				strength_mask |= 1<<s;
				stronger_bb |= current_bb;
			}
		}
		for(int s=0;s<=Strongest;s++) {
			if ((strength_mask & (1<<s)) != 0) {
				usedPieceNames += InclPieceNames.substring(s << 1, (s + 1) << 1);
			} else {
				unusedPieceNames += InclPieceNames.substring(s << 1, (s + 1) << 1);
			}
		}
		String newInclPieceNames = usedPieceNames+unusedPieceNames+ori_unusedPieceNames;
		return ChangeRules(newInclPieceNames,(usedPieceNames.length() >> 1)-1);
	}

	/**
	 * Searches ori_move_list (stings in official notation) for last capture, and sets this to the position just after the capture and  
	 * returns move_list starting after the capture. GameState (and this) is compressed afterwards.
	 * So move_list is shortened to sufficient portion for 3(2) fold repetition tests and the Strongest is set to appropriate size.
	 * Compress of GameState is compatible with ZORB hashing so the hash table entries would remain accessible.
	 * Position in material graph is updated after compression so movement in the graph works well afterwards as well.
	 */
	
	/**
	 * Constructor providing given rules would apply (number of remaining pieces)
	 * The material expects there is never more pieces than in standard arimaa setup.
	 * @param rules
	 */
	public GameState(int rules) {
		if (rules==0) {
			if (InclPieceNames.length()!=12) {
				InclPieceNames = "RrCcDdHhMmEe";
				Strongest = 5;
				Init();
			} else {
				Init();
				ChangeRules("RrCcDdHhMmEe",5);
			}
			// piece counts "882222221111"
			turn = 1;
			player = 0;
			enemy = 1;
			steps_remaining = 4;
		} else {
			// for rules with more pieces Zobrist and material must be allocated accordingly
			LogFile.message("Unknown rules");
		}
	}

	/** Names of pieces of the current position (remaining ordered by strength, pieces already captured are at the end.
	 */
	public static String InclPieceNames="RrCcDdHhMmEe";
	// pieces included in starting position either cat forced Cc ...
	/**
	 * Relative strength of the strongest remaining piece on the board (number of different pieces-1)
	 */
	public static int Strongest=5;
	/** Zobr table for the pieces and places on the board to be used for hashing
	 */
	public static long Zobr[][][] = new long[6][2][64];
	// just [Strongest+1][2][64] would be used ... order of dimensions
	// prepared to allow shrinking of piece strengths
	/**
	 * Zobr table to represent who's step it is and which
	 */
	public static long ZobrSteps[] = new long[8];
	// [(player<<2) ^ steps_remaining]
	static {
		HashSet<Long> zorbKeys = new HashSet<Long>();
		for (int i = 0; i < 8; i++) {
			long r;
			while (true) {
				r=random.nextLong();
				int popCnt=Util.PopCnt(r);
				if (popCnt<29) {
					// System.out.println("Too few bits for Zorbing "+Long.toHexString(r)+" skipped");
					continue;
				}
				if (popCnt>35) {
					// System.out.println("Too many bits for Zorbing "+Long.toHexString(r)+" skipped");
					continue;
				}
				if (zorbKeys.contains(r)) {
					// System.out.println("Duplicate random number for Zorbing "+Long.toHexString(r)+" skipped");
					continue;
				}
				break;
			}
			zorbKeys.add(r);
			ZobrSteps[i] = r ;
		}
		// long doubleRankMasks[] = {ZorbGoldHomeMask,ZorbMiddleMask,ZorbMiddleMask,ZorbSilverHomeMask};
		//int popCntBounds[][] = new int[4][2];
		//for (int i=0;i<4;i++) {
		//	popCntBounds[i][0]=(Util.PopCnt(doubleRankMasks[i])>>1)-3;
		//	popCntBounds[i][1]=(Util.PopCnt(doubleRankMasks[i])>>1)+3;
		//}
		for (int index = 0; index < 64; index++) {
			for (int color = 0; color < 2; color++) {
				for (int strength = 0; strength < 6; strength++) {
					long r;
					while (true) {
						r=random.nextLong();
						//r=random.nextLong() & doubleRankMasks[index>>4];
						int popCnt=Util.PopCnt(r);
						//if (popCnt<popCntBounds[index>>4][0]) {
						if (popCnt<29) {
							// System.out.println("Too few bits for Zorbing "+Long.toHexString(r)+" skipped");
							continue;
						}
						//if (popCnt>popCntBounds[index>>4][1]) {
						if (popCnt>35) {
							// System.out.println("Too many bits for Zorbing "+Long.toHexString(r)+" skipped");
							continue;
						}
						if (zorbKeys.contains(r)) {
							// System.out.println("Duplicate random number for Zorbing "+Long.toHexString(r)+" skipped");
							continue;
						}
						break;
					}
					zorbKeys.add(r);
					Zobr[strength][color][index] = r;
				}
			}
		}
	}

	/**
	 * color of the player to move
	 */
	public int player;
	
	/**
	 * color of the player to wait
	 */
	public int enemy;
	/**
	 * Number of steps allowed to be played yet in this turn
	 */
	public int steps_remaining;
	
	/**
	 * Turn number of the game 
	 */
	public int turn;
	
	/**
	 * Zobr hash of the current position excluding the ZobrSteps part
	 */
	public long zobr_hash;
	
	/**
	 * Representation of the material included in the position
	 */
	public Material mat; // material and it's evaluation
	
	/**
	 * attempt to deviate during the game thanks to bounded randomness
	 */
	public int rand_score=0;

	// piece_type starts from 2 to ((Strongest<<1)|1)
	/**
	 * Strengths 00 Rabbits BB (unless game is already finished) 01 1st stronger
	 * than rabbit (Cc ?) 02 2nd stronger than rabbit (Dd ?) 03 3rd stronger
	 * than rabbit (Hh ?) 04 4th stronger than rabbit (Mm ?) 05 5th stronger
	 * than rabbit (Ee if every strength present)
	 * 
	 * Player (Color) ID's are as follows 00 White 01 Black
	 * 
	 * Piece ID is (strongID<<1 | Color)
	 */

	/**
	 * The basic information
	 */
	public long stronger_or_eq_bb[];
	
	/**
	 * Together with the information ... which pieces belong to player 0 an which to player 1
	 */
	public long player_bb[] = new long[2];
	// for [Strongest+2] ... the last is always 0

	/**
	 *  which part of the board remains empty empty_bb | player_bb[0] | player[1] is entire board, each bit just once.
	 */
	public long empty_bb;

	/** field required when generating steps so computed as secondary_bitboards in playFull routine
	 */
	public long frozen_pieces_bb;
	/** needed in frozen_pieces_bb computation, and in gencaptures as well used in evaluation (so by playFull)
	 */
	public long dominated_pieces_bb;
	/** needed in frozen_pieces_bb computation, and in gencaptures as well used in evaluation (so by playFull)
	 */
	public long no_friend_bb; // piece not touched by a friend
	/**
	 * Usefull in evaluation squares with no empty neighbours
	 */
	public long crosses_bb;
	
	/** piece touched by at most one friend ... tertiary info 
	 */
	public long at_most_one_friend_bb;
	/** pieces which cannot move ... tertiary info not computed well so far (approximated by forzen)
	 */
	public long passive_pieces_bb; 
	// cannot slide or push except ending in trap ... frozen or blocked ... blocked by phalanx or not weaker piece
	// phalanx could be multicolored, can be touched by at most one piece of the same color
	// trap could be guarded so piece could escape through trap
	// either good hostage or frozen piece or blocked piece
	/** pieces dominated by one stronger pieces ... tertiary info used in evaluation
	 */
	public long barely_dominated_pieces_bb;
	
	//public long[] hanging_bb = {0,0,0}; // by steps to capture 2,3,4 steps
	//public long hostaged_bb; //frozen, can be unfrozen just to run near the unprotected trap
	//public long frozen_to_capture_bb; // ready for immediate capture ... trap defender required/unfreeze and escape
	//public long hostage_to_capture_bb; // ready for immediate capture ... trap defender required, no escape/unfreeze possible
	
	// Working variables for genMove routines
	/** the lone defender of a piece in trap secondary info
	 */
	public long trap_protect_bb; 
	/** traps with at most one color piece touching secondary info 
	 */
	public long suicide_bb[] = new long[2];
	/** Location of single piece protecting trap (even when not holding piece) secondary info
	 */
	public long lone_defender_bb[] = new long[2];
	/** bit set if the place has at least that number of players neighbors
	 */
	public long count_touch_bb[][] = new long[2][5];
	
	/** board to retrieve pieces at given position ... primary info
	 */
	public int piece_at[] = new int[64];
	
	/** translation from piece name to piece int (according to current rules)
	 * @param piece_name
	 * @return strength<<1|color
	 */
	public static int Name2InclPiece(String piece_name) {
		return InclPieceNames.indexOf(piece_name);
	}

	/** transformation from piece int to piece name (according to current rules)
	 * @param piece_type (strength<<1|color)
	 * @return piece_name
	 */
	public static String InclPiece2Name(int piece_type) {
		assert (piece_type >= -1 && piece_type <= StrengthAndColor2InclPiece(
				Strongest, 1));
		if (piece_type>=0) return InclPieceNames.substring(piece_type, piece_type + 1);
		return " ";
	}

	public static String StrengthAndColor2Name(int strength, int color) {
		return InclPiece2Name(StrengthAndColor2InclPiece(strength, color));
	}

	public static int StrengthAndColor2InclPiece(int strength, int color) {
		return (strength << 1) | color;
	}

	public static int InclPiece2Strength(int piece_type) {
		return piece_type >> 1;
	}

	public static int InclPiece2Color(int piece_type) {
		return piece_type & 1;
	}

	public static int Name2Strength(String name) {
		return Name2InclPiece(name) >> 1;
	}

	public static int Name2Color(String name) {
		return Name2InclPiece(name) & 1;
	}

	/**
	 * mask variant is required for moves Get the piece type of a square
	 * 
	 * @param index
	 *            int
	 * @return int
	 */
	public int getPieceType(int index) {
		int ret=piece_at[index];
		if (testing) {
			if (ret==-1) {
				if ((empty_bb&(1L<<index))==0) {
					int debug_breakpoint_here=1;debug_breakpoint_here++;					
				}
			} else if (((stronger_or_eq_bb[ret>>1]^stronger_or_eq_bb[1+(ret>>1)])&(1L<<index))==0) {
				int debug_breakpoint_here=1;debug_breakpoint_here++;
			}
		}
		return ret;
	}

	public int getPieceTypeOneSq(long one_bb) {
		return getPieceType(Util.bit2ind(one_bb));
	}
	
	/**
	 * Returns the piece type of the target bb if more bit's set, finds
	 * strongest piece (and prefers silver). usually called with just one bit
	 * set
	 * 
	 * @param target_bb
	 *            long
	 * @return int (-1 represents empty squares)
	 */
	public int getPieceType(long target_bb) {
		return Math.max(getPieceType(target_bb, 0), getPieceType(target_bb, 1));
	}

	/**
	 * returns the strongest piece of given color on target_bb -1 represents
	 * none
	 */
	public int getPieceType(long target_bb, int color) {
		int s = getStrongest(target_bb, color);
		if (s == -1) {
			return s;
		}
		return StrengthAndColor2InclPiece(s, color);
	}

	public int getStrongestDn(long target_bb, int color, int s) {
		return getStrongestDn(target_bb &= player_bb[color],s);
	}
	
	public int getStrongestDn(long target_bb, int s) {
		// there is an option to try binary search, but the sizes of
		// stronger_or_eq_bb sets usually decrease almost exponencially
		// one could test "dogs + cats" prior to cats, but that one test is not
		// worth it
		// (ER only boards would need exception in such code)
		assert(target_bb != 0);
		while ((stronger_or_eq_bb[s] & target_bb) == 0) {
			s--;
		}
		return s;
	}

	/**
	 * returns the strength of the strongest piece of given color on target_bb -1
	 * represents none
	 */
	public int getStrongest(long target_bb, int color) {
		return getStrongest(target_bb &= player_bb[color]);
	}

	/**
	 * returns the strength of the strongest piece on target_bb -1 represents none
	 */
	public int getStrongest(long target_bb) {
		int s = 0; 
		// there is an option to try binary search, but the sizes of
		// stronger_or_eq_bb sets usually decrease almost exponencially
		// one could test "dogs + cats" prior to cats, but that one test is not
		// worth it
		// (ER only boards would need exception in such code)
		while ((stronger_or_eq_bb[s] & target_bb) != 0) {
			s++;
		}
		return s-1;
	}
	/**
	 * returns the strength of the weakest piece on target_bb -1 represents none
	 */
	public int getWeakestPieceStrength(long target_bb) {
		target_bb &= ~empty_bb;
		if (target_bb == 0)
			return -1;
		int s = 1;
		while ((target_bb & stronger_or_eq_bb[s]) == target_bb) {
			s++;
		}
		return s-1;
	}

	public void copy(GameState orig) {// could copy to itself
		// copies what secondary_bitboars calculates
		if (this==orig) return; // to speed up special case ;)
		player = orig.player;
		enemy = orig.enemy;
		steps_remaining = orig.steps_remaining;
		zobr_hash = orig.zobr_hash;
		turn = orig.turn;
		mat = orig.mat;
		for(int p=0; p<2; p++) {
			player_bb[p] = orig.player_bb[p];
		}
		for (int s = 0; s <= Strongest+1; s++) {
			stronger_or_eq_bb[s] = orig.stronger_or_eq_bb[s];
		}
		for (int i = 0; i<64; i++) {
			piece_at[i]=orig.piece_at[i];
		}
		empty_bb = orig.empty_bb;
		crosses_bb = orig.crosses_bb;
		no_friend_bb=orig.no_friend_bb;
		frozen_pieces_bb = orig.frozen_pieces_bb;
		dominated_pieces_bb = orig.dominated_pieces_bb;
		barely_dominated_pieces_bb = orig.barely_dominated_pieces_bb;
		trap_protect_bb = orig.trap_protect_bb;
		for (int i=0;i<2;i++) {
			suicide_bb[i] = orig.suicide_bb[i];
			lone_defender_bb[i] = orig.lone_defender_bb[i];
			for (int n=0;n<5;n++) {
				count_touch_bb[i][n]=orig.count_touch_bb[i][n];
			}
		}
		passive_pieces_bb=orig.passive_pieces_bb;
	}

	/**
	 * Returns true if all pieces are in same position. Side to move and steps
	 * remaining do not matter
	 * 
	 * @param that GameState
	 * @return boolean
	 */
	public boolean equals(GameState that) {
		if (this.zobr_hash != that.zobr_hash) {
			return false;
		}
		// now with very high probability the positions are equal. Let us check
		// that.
		for (int p = 0; p < 2; p++) {
			if (this.player_bb[p] != that.player_bb[p]) {
				return false;
			}
		}
		for (int s = 0; s <= Strongest; s++) {// [c][Strongest+1]==0
			if (this.stronger_or_eq_bb[s] != that.stronger_or_eq_bb[s]) {
				return false;
			}
		}
		return true;
	}

	private void checkZobr() {
		long hash=0;
		for(int i=0;i<64;i++) {
			int piece=this.getPieceType(i);
			if (piece != -1) {
				int strength = InclPiece2Strength(piece);
				int color = InclPiece2Color(piece);
				hash ^= Zobr[strength][color][i];
			}
		}
		if (hash != zobr_hash) {
			LogFile.message("Zobr error " + this + hash + "#" + zobr_hash);
		}		
	}
	
	/**
	 * Gets hash code for entire position Uses zobrist for steps remaining and
	 * side to move
	 * 
	 * @return long
	 */
	public long getPositionHash() {
		if ((((player << 2) ^ steps_remaining)>8)||(((player << 2) ^ steps_remaining)<0)) {
			int debug=0;
			return debug ^ zobr_hash ^ ZobrSteps[(player << 2) ^ steps_remaining];			
		}
		return zobr_hash ^ ZobrSteps[(player << 2) ^ steps_remaining];
	}

	/**
	 * Returns true if all pieces in same position and number of steps the same
	 * @param that GameState
	 * @return boolean
	 */
	public boolean full_equals(GameState that) {
		if ((this.player != that.player)
				|| (this.steps_remaining != that.steps_remaining)) {
			return false;
		}
		return this.equals(that);
	}

	public void playPASSturn() {
		//steps_remaining = 4;
		turn += player;
		player ^= 0x01;
		enemy ^= 0x01;
	}

	public void unplayPASS() {
		//steps_remaining = 4;
		enemy ^= 0x01;
		player ^= 0x01;
		turn -= player;
	}

	// Allocates object arrays (Strongest is known)
	// empty board with InclPieceNames allowed
	private void Init() {
		stronger_or_eq_bb = new long[Strongest + 2];
		for (int strength = 0; strength <= Strongest + 1; strength++) {
			stronger_or_eq_bb[strength] = 0L;
		}
		empty_bb = -1L;
		for (int i = 0; i<64; i++) {
			piece_at[i]=-1;
		}
		zobr_hash = 0L;
		mat = null;
		player=0; enemy=1;
	}

	// Creates the position based on the setup text strings that start the games
	// sets global variables according to "all pieces present"
	public GameState(String white_setup_text, String black_setup_text) {

		player = 0;
		enemy = 1;
		turn = 2;
		steps_remaining = 4;

		Init();
		ChangeRules("RrCcDdHhMmEe",5);
		String full = white_setup_text + " " + black_setup_text;
		StringTokenizer tokenizer = new StringTokenizer(full);

		// first run to detect included strengths (for puzzles these need not be
		// all)
		while (tokenizer.hasMoreTokens()) {
			String piece_token = tokenizer.nextToken();

			// Ignore move count tokens
			if (piece_token.equals("1w") || piece_token.equals("1b")) {
				continue;
			}

			if (piece_token.equals("2w")) {// second turn must be handled
											// another way
				break;
			}
		}

		tokenizer = new StringTokenizer(full);

		while (tokenizer.hasMoreTokens()) {
			String piece_token = tokenizer.nextToken();

			// Ignore move count tokens
			if (piece_token.equals("1w") || piece_token.equals("1b")) {
				continue;
			}

			if (piece_token.equals("2w")) {// second turn must be handled
											// another way
				break;
			}

			// Convert the text move to numbers
			int piece_type = Name2InclPiece(piece_token.substring(0, 1));
			int color = InclPiece2Color(piece_type);
			int strength = InclPiece2Strength(piece_type);
			int col = Notation.Name2Col(piece_token.substring(1, 2));
			int row = Notation.Name2Row(piece_token.substring(2, 3));
			// Populate the bitboard
			int bb_index = Notation.RowCol2Index(row, col);
			piece_at[bb_index]=piece_type;
			zobr_hash ^= Zobr[strength][color][bb_index];
			long mask = 1L << bb_index;
			player_bb[color] |= mask;
			for (int s = strength; s >= 0; s--) {
				stronger_or_eq_bb[s] |= mask;
			}
		}

		empty_bb = ~(stronger_or_eq_bb[0]);
		// 0x0000ffffffff0000 for initial move, but not for puzzles
		assert (trapsOK());
		mat = Material.getMaterial(this);
		compute_secondary_bitboards();
		// this position would be searched ...
		Compress(); 
		// for puzzle setups not using all the pieces
	}

	/**
	 * Constructs the game state from text string Format is same as used by bot
	 * interface program
	 * 
	 * @param text String
	 */
	public GameState(String text) {

		Init();
		int space_pos = text.indexOf(" ", 0);
		int ppos = text.indexOf("%", 0);

		String turn_text = text.substring(0, space_pos - 1);
		String player_text = text.substring(space_pos - 1, space_pos);
		// String move_text = text.substring(space_pos, ppos);
		String board_text = text.substring(ppos, text.length());

		// Determine total steps
		turn = Integer.parseInt(turn_text);
		player = (player_text.equals("w")) ? 0 : 1;
		enemy = player ^ 0x01;
		steps_remaining = 4;

		ChangeRules("RrCcDdHhMmEe",5);

		// Populate the piece bitboards
		for (int row = 7; row >= 0; row--) {
			for (int col = 0; col <= 7; col++) {
				int bb_index = Notation.RowCol2Index(row, col);
				int text_index = 29 + 2 * col + (7 - row) * 23;
				String letter = board_text
						.substring(text_index, text_index + 1);
				int piece_type = Name2InclPiece(letter);
				piece_at[bb_index]=piece_type;
				int color = InclPiece2Color(piece_type);
				int strength = InclPiece2Strength(piece_type);
				if (strength>=0) {
					zobr_hash ^= Zobr[strength][color][bb_index];
					long mask = 1L << bb_index;
					player_bb[color] |= mask;
					for (int s = strength; s >= 0; s--) {
						stronger_or_eq_bb[s] |= mask;
					}
				}
			}
		}

		empty_bb = ~(stronger_or_eq_bb[0]);
		assert (trapsOK());
		assert (stronger_or_eq_bb[Strongest+1]==0);
		mat = Material.getMaterial(this);
		compute_secondary_bitboards();
		// this position would be searched ...
		Compress();
	}

	/** checks the position consistency ... no piece in trap without a friend
	 * does not change the position
	 * @return true if OK
	 */
	public boolean trapsOK() {
		return (no_friend_bb&TRAP_SQUARES)==0;
	}

	/**
	 * Determines all potential suicides for both colours
	 */
	void calculate_trap_info() {
		trap_protect_bb = 0L;
		for (int c = 0; c < 2; c++) {
			neighbourSum(player_bb[c], count_touch_bb[c]);
			suicide_bb[c] = TRAP_SQUARES & (~count_touch_bb[c][2]);
			trap_protect_bb |= player_bb[c] & touching_bb(suicide_bb[c] & player_bb[c]);
			lone_defender_bb[c] = player_bb[c] & touching_bb(suicide_bb[c]);
		}
	}

	private void log_bitboards() {
		LogFile.writeln(toString());
		for (int s = 0; s <= Strongest; s++) {
			LogFile.writeln(">="+InclPiece2Name(s<<1)+bitboard(stronger_or_eq_bb[s]));
		}
		for(int c = 0; c < 2; c++) {
			LogFile.writeln("color "+c+bitboard(player_bb[c]));			
		}
		LogFile.writeln("empty"+bitboard(empty_bb));
	}
	
	private void consistency_Check() throws MaterialInconsistencyException{
		// not checking the bitboard inclusions
		for(int ind = 0; ind < 64; ind++) {
			long bb=(1L<<ind);
			if (piece_at[ind]>=0) {
				int player = piece_at[ind]&1, opponent = 1-player;
				int s = piece_at[ind]>>1;
				if ((player_bb[player]&bb)==0) {
					throw new MaterialInconsistencyException("Player not on "+Notation.Index2Name(ind)+" ");
				}
				if ((player_bb[opponent]&bb)!=0) {
					throw new MaterialInconsistencyException("Opponent on "+Notation.Index2Name(ind)+" ");
				}
				if ((empty_bb&bb)!=0) {
					throw new MaterialInconsistencyException("Empty on "+Notation.Index2Name(ind)+" ");
				}
				if (((stronger_or_eq_bb[s+1])&bb)!=0) {
					throw new MaterialInconsistencyException("Stronger on "+Notation.Index2Name(ind)+" ");
				}
				if (((stronger_or_eq_bb[s])&bb)==0) {
					throw new MaterialInconsistencyException("Piece not on "+Notation.Index2Name(ind)+" ");
				}
			} else {
				if ((player_bb[0]&bb)!=0) {
					throw new MaterialInconsistencyException("Gold on empty "+Notation.Index2Name(ind)+" ");
				}
				if ((player_bb[1]&bb)!=0) {
					throw new MaterialInconsistencyException("Silver on empty "+Notation.Index2Name(ind)+" ");
				}
				if ((empty_bb&bb)==0) {
					throw new MaterialInconsistencyException("Notempty on empty "+Notation.Index2Name(ind)+" ");
				}				
				if ((stronger_or_eq_bb[0]&bb)!=0) {
					throw new MaterialInconsistencyException("A piece on empty "+Notation.Index2Name(ind)+" ");
				}
			}
		}
		if (stronger_or_eq_bb[Strongest+1]!=0) {
			throw new MaterialInconsistencyException("Stronger than strongest exists");
		}
		long bb=0;
		for(int s=Strongest;s>=0;s--) {
			if ((stronger_or_eq_bb[s]&bb)!=bb) {
				throw new MaterialInconsistencyException("inclusion by strength is corrupted "+s);
			}
			bb=stronger_or_eq_bb[s];
		}
	}
	/**
	 * Computes bitboards usefull for step generation ... frozen_pieces_bb and crosses_bb
	 * Usefull sub results are saved as well
	 */
	public void compute_secondary_bitboards() {
		//try {
		//	inconsistent = false;
		//	consistency_Check();
		//} catch (MaterialInconsistencyException b) {
		//	inconsistent = true;
		//	LogFile.write("Bug: Inconsistent position "+b.getMessage());
		//	log_bitboards();
		//	b.logStackTrace();
		//}
		calculate_trap_info();// calculates count_touch_bb[][]
		crosses_bb = ~touching_bb(empty_bb);
		no_friend_bb=(player_bb[0]&~count_touch_bb[0][1])|
			(player_bb[1]&~count_touch_bb[1][1]);
		dominated_pieces_bb = 0L; // All dominated pieces
		for (int s = 1; s <= Strongest; s++) {
			dominated_pieces_bb |= player_bb[1] & touching_bb(stronger_or_eq_bb[s]&player_bb[0]) & ~stronger_or_eq_bb[s];
			dominated_pieces_bb |= player_bb[0] & touching_bb(stronger_or_eq_bb[s]&player_bb[1]) & ~stronger_or_eq_bb[s];
		}
		frozen_pieces_bb = dominated_pieces_bb & no_friend_bb;
	}

	/**
	 * Computes bitboards usefull for static evaluation
	 */
	public void compute_tertiary_bitboards() {
		at_most_one_friend_bb = (player_bb[0]&~count_touch_bb[0][2])|
			(player_bb[1]&~count_touch_bb[1][2]);
		barely_dominated_pieces_bb = 0L; // All dominated by just stronger piece
		passive_pieces_bb = at_most_one_friend_bb & (~touching_bb(empty_bb & NON_TRAP_SQUARES));// no square to slide
		// TODO test if it could push away and lately or it with frozen_pieces_bb, rabbits cannot move backwards ...
		long s_bb=stronger_or_eq_bb[0] ^ stronger_or_eq_bb[1], s_bbp;
		for (int s = 1; s <= Strongest; s++) {
			s_bbp = stronger_or_eq_bb[s] ^ stronger_or_eq_bb[s+1];
			barely_dominated_pieces_bb |= player_bb[1] & touching_bb(s_bbp & player_bb[0]) & s_bb;
			barely_dominated_pieces_bb |= player_bb[0] & touching_bb(s_bbp & player_bb[1]) & s_bb;
			s_bb = s_bbp;
		}
		passive_pieces_bb |= frozen_pieces_bb;
	}

	//								gold goal		= 0xffff0000
	//								silver goal		= 0xff00ff00
	//								last silver		= 0xf0f0f0f0
	//								steps_remain & 3 ... last 2 bits
	private static final long table_is_game_over	= 0xff1ff100;	
	private static final long table_enemy_won 		= 0x00100100;
	// This table tells if the game is over by rabbit goal or elimination
	// An enemy rabbit can be in the goal, if there are steps remaining!

	private static int WN = +SCORE_MATE;
	private static int LS = -SCORE_MATE;

	/**
	 * Considers only current gamestate, no previous move required
	 * 
	 * @return boolean
	 * called after the move is played so the player is already changed !!!
	 */

	public boolean isGameOver() {
		// if ((mat.piece_count[0][0]==0) || (mat.piece_count[1][0]==0)) return
		// true;
		// if TestForGoal.test() works well, this will be used only when
		// searching for the last move of the game
		long r_bb = stronger_or_eq_bb[0] ^ stronger_or_eq_bb[1];
		long gr_bb = r_bb & player_bb[0];
		long sr_bb = r_bb & player_bb[1];
		int gold_goal = (((gr_bb & RANK_8) != 0) || (sr_bb == 0)) ? 1 : 0;
		int silver_goal = (((sr_bb & RANK_1) != 0) || (gr_bb == 0)) ? 1 : 0;

		int index = (gold_goal << 4) | (silver_goal << 3) | (player << 2)
				^ steps_remaining;
		// steps_remaining 4,3,2,1 (if 3,2,1,0 the table_is_game_over would
		// remain the same, but table_enemy_won would be 0x00100100)

		return (table_is_game_over & (1L << index)) != 0;
	}

	/**
	 * Returns result of game SCORE_MATE,-SCORE_MATE From perspective of side to
	 * move at start of *previous* step
	 * called after the move is played so the player is already changed !!!
	 * @return int
	 */
	public int getGameResult() {
		assert (isGameOver() == true);

		// Test for rabbit goal / all rabbit capture
		long r_bb = stronger_or_eq_bb[0] ^ stronger_or_eq_bb[1];
		long gr_bb = r_bb & player_bb[0];
		long sr_bb = r_bb & player_bb[1];
		int gold_goal = (((gr_bb & RANK_8) != 0) || (sr_bb == 0)) ? 1 : 0;
		int silver_goal = (((sr_bb & RANK_1) != 0) || (gr_bb == 0)) ? 1 : 0;
		int index = (gold_goal << 4) | (silver_goal << 3) | (player << 2)
				^ steps_remaining;
		long mask = 1L << index;
		assert ((table_is_game_over & mask) != 0);
		return ((table_enemy_won & mask) != 0) ? LS : WN;
	}

	/**
	 * Returns the side to move
	 * @return int
	 */
	public int getSideToMove() {
		return player;
	}

	/**
	 * Returns the turn number
	 * @return int
	 */
	public int getTurnNumber() {
		return turn;
	}

	/**
	 * Returns the steps remaining in the current turn (4 represents 0)
	 * @return int
	 */
	public int getStepsRemaining() {
		return steps_remaining;
	}

	/**
	 * Returns current position as the first two moves of the game. This can be
	 * pasted into the planning applet located at
	 * http://arimaa.com/arimaa/games/planGame.cgi
	 * 
	 * @return String
	 */
	public String toSetupString() {
		String result = "";

		// Get all the white pieces
		result += "1w ";

		for (int s = Strongest; s >= 0; s--) {
			long temp_bb = stronger_or_eq_bb[s] ^ stronger_or_eq_bb[s + 1];
			temp_bb &= player_bb[0];
			while (temp_bb != 0) {
				long lsb_bb = Util.LastBit(temp_bb);
				temp_bb ^= lsb_bb;
				int index = Util.FirstOne(lsb_bb);
				result += StrengthAndColor2Name(s, 0);
				result += Notation.Index2Name(index);
				result += " ";
			}
		}
		result += "\n";

		// Get all the black pieces
		result += "1b ";

		for (int s = Strongest; s >= 0; s--) {
			long temp_bb = stronger_or_eq_bb[s] ^ stronger_or_eq_bb[s + 1];
			temp_bb &= player_bb[1];
			while (temp_bb != 0) {
				long lsb_bb = Util.LastBit(temp_bb);
				temp_bb ^= lsb_bb;
				int index = Util.FirstOne(lsb_bb);
				result += StrengthAndColor2Name(s, 1);
				result += Notation.Index2Name(index);
				result += " ";
			}
		}
		result += "\n";

		// Add pass move if its black's turn
		if (this.player == 1) {
			result += "2w pass\n";
		}

		return result;
	}

	/**
	 * Returns current gamestate as a string
	 * 
	 * @return String
	 */
	public String toString() {
		String result = "";
		// Make first line
		result += getTurnNumber();
		result += (getSideToMove() == PL_GOLD) ? "w" : "b";

		// add DUMMY move to account for any partial moves played ???
		for (int i = 4; i > steps_remaining; i--) {
			result += " xxxx";
		}

		// display the board
		result += " \n +-----------------+\n";
		for (int row = 7; row >= 0; row--) {
			result += Notation.Row2Name(row) + "|";

			for (int col = 0; col <= 7; col++) {
				int board_index = Notation.RowCol2Index(row, col);
				result += " " + InclPiece2Name(getPieceType(board_index));
			}
			result += " |\n";

		}
		result += " +-----------------+\n";
		result += "   a b c d e f g h\n";

		result += "SR: " + steps_remaining + "\n";
		if (mat!=null) {
			result += "MAT: " + mat + "\n";
		} else {
			result += "MAT: invalid\n";
		}
		result += "Z: " + zobr_hash + "\n";
		result += "RS: " + rand_score + "\n:";
		return result;
	}

	/**
	 * Returns the current position as a board string
	 * @return String
	 */
	public String toBoardString() {
		String text = this.toString();
		return text;
	}

	/**
	 * Returns the current position as an EPD string
	 * 
	 * @return String
	 */
	public String toEPDString() {
		String text = this.toString();
		text = text.replaceAll("\n", "%13");
		return text;
	}

	/**
	 * Also computes the secondary bitboards (needed for following steps selection). This is a lot slower than unplayMove() on its own.
	 * @param move ArimaaMove
	 */

	public void unplayMoveFull(ArimaaMove move) {
		//LogFile.message("Unplaying "+move);
		//LogFile.message("From "+this);
		unplayMove(move);
		//LogFile.message("To "+this);
		compute_secondary_bitboards();
		if (inconsistent) {
			LogFile.message("unplayed move "+move);
			for(int s=0;s<=Strongest;s++) {
				LogFile.message("move["+s+"]from "+bitboard(move.from_bb[s]));
				LogFile.message("move["+s+"]to "+bitboard(move.to_bb[s]));
			}
			LogFile.message("enemyfrom"+bitboard(move.enemy_from_bb));
			LogFile.message("enemyto"+bitboard(move.enemy_to_bb));
			LogFile.message("StackTrace"+MaterialInconsistencyException.stackTrace());
		}
	}

	public void unplayMove(ArimaaMove move) {//could unplay full turn or just slide or drag but not the setup move
		if (steps_remaining == 4) {
			player ^= 0x01;
			enemy ^= 0x01;
			steps_remaining = 0;
			turn -= player;
		}
		if ((player==1)==((move.info_mask&ArimaaMove.SILVER_MARK)==0)) {
			LogFile.message("Bug ... unmove does not correspond to the player (turncount already updated)" + move + this);
			LogFile.message("StackTrace"+MaterialInconsistencyException.stackTrace());
		}
		steps_remaining = steps_remaining + move.steps;
		zobr_hash ^= move.zobr_hash;

		player_bb[enemy] ^= move.enemy_from_bb ^ move.enemy_to_bb;
		// for player ... it must be done per places moved
		try {
			mat = mat.uncapture(move.info_mask & ArimaaMove.CAPTURE_COUNTS);
		} catch (MaterialInconsistencyException b) {
			LogFile.writeln("Material inconsistency unplaying "+move+"in position "+this);
			b.logStackTrace();
		}
		
		long moved_stronger_or_eq_bb = 0L;
		for (int s = Strongest; s >= 0; s--) {// player move first
			stronger_or_eq_bb[s] ^= (moved_stronger_or_eq_bb ^= (move.from_bb[s]^move.to_bb[s]));
		}
		for (int s = 0; s <= Strongest; s++) {
			long temp_bb;
			if (move.to_bb[s]!=0) {
				if ((temp_bb=move.to_bb[s] & move.enemy_to_bb)!=0) {
					for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
						lsb_bb=Util.LastBit(temp_bb);
						int ind=Util.bit2ind(lsb_bb);
						piece_at[ind]=-1;
					}
				}
				if ((temp_bb=move.to_bb[s] & ~move.enemy_to_bb)!=0) {
					player_bb[player] ^= temp_bb;
					for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
						lsb_bb=Util.LastBit(temp_bb);
						int ind=Util.bit2ind(lsb_bb);
						piece_at[ind]=-1;
					}
				}
			}
		}
		for (int s = 0; s <= Strongest; s++) {
			long temp_bb;
			if (move.from_bb[s]!=0) {
				if ((temp_bb=move.from_bb[s] & move.enemy_from_bb)!=0) {
					int piece=StrengthAndColor2InclPiece(s,enemy);
					for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
						lsb_bb=Util.LastBit(temp_bb);
						int ind=Util.bit2ind(lsb_bb);
						piece_at[ind]=piece;
					}
				}
				if ((temp_bb=move.from_bb[s] & ~move.enemy_from_bb)!=0) {
					player_bb[player] ^= temp_bb;
					int piece=StrengthAndColor2InclPiece(s,player);
					for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
						lsb_bb=Util.LastBit(temp_bb);
						int ind=Util.bit2ind(lsb_bb);
						piece_at[ind]=piece;
					}
				}
			}
		}
		empty_bb = ~(stronger_or_eq_bb[0]);
		rand_score = ((int) (move.info_mask & ArimaaMove.RAND_SCORE_MASK)) >> 28;
	}
	
	/**killer may be inconsistent with the position this does not try to verify its correctness
	 * so player pieces could be frozen on the way and pieces lost on guarded traps or stayed in unguarded traps
	 * @param killer
	 * @return
	 */
	public boolean isKillerConsistent(ArimaaMove killer) {
		assert (steps_remaining >= killer.steps);
		long asempty_bb=empty_bb, all_to_bb=0L;
		for (int s = 0; s <= Strongest; s++) {// does the pieces start at correct places? Would there be space to move to?
			if (killer.from_bb[s]!=0) {
				all_to_bb |= killer.to_bb[s];
				asempty_bb|=killer.from_bb[s];
				long test_bb;
				if ((test_bb=killer.from_bb[s]&killer.enemy_from_bb)!=0) {
					if ((player_bb[enemy]&test_bb)!=test_bb) {
						return false; // wrong player
					}
				}
				if ((test_bb=killer.from_bb[s]&~killer.enemy_from_bb)!=0) {
					if ((player_bb[player]&test_bb)!=test_bb) {
						return false; // wrong player
					}
				}
				if (((stronger_or_eq_bb[s]^stronger_or_eq_bb[s+1])&killer.from_bb[s])!=killer.from_bb[s]) {
					return false; // wrong strength
				}
			}
		}
		if ((all_to_bb & ~asempty_bb)!=0) {//step to occupied square
			return false;
		}
		return true;
	}
	
	/**
	 * Also computes the secondary bitboards (needed for following steps selection). This is a lot slower than playMove() on its own.
	 * @param move ArimaaMove
	 */

	public void playMoveFull(ArimaaMove move) {
		playMove(move);
		compute_secondary_bitboards();
		if (inconsistent) {
			LogFile.message("played move "+move);
			for(int s=0;s<=Strongest;s++) {
				LogFile.message("move["+s+"]from "+bitboard(move.from_bb[s]));
				LogFile.message("move["+s+"]to "+bitboard(move.to_bb[s]));
			}
			LogFile.message("enemyfrom"+bitboard(move.enemy_from_bb));
			LogFile.message("enemyto"+bitboard(move.enemy_to_bb));
			LogFile.message("StackTrace"+MaterialInconsistencyException.stackTrace());
		}
	}

	/**
	 * Plays the given move on the provided previous gamestate
	 * @param move ArimaaMove
	 * @param prev GameState be carefull this could be equal prev!
	 */
	public void playMove(ArimaaMove move) {
		//if (steps_remaining < move.steps) {
		//	LogFile.message("steps remaining" + steps_remaining + "move steps" + move.steps+this+move);
		//}
		assert (steps_remaining >= move.steps);
		if (testing) {		
			if ((player==1)==((move.info_mask&ArimaaMove.SILVER_MARK)==0)) {
				LogFile.message("Bug ... move does not correspond to the player" + move + this);
				LogFile.message("StackTrace"+MaterialInconsistencyException.stackTrace());
			}
			if (!isKillerConsistent(move)) {
				int breakpoint_here=0; breakpoint_here++;
			}
		}
		long moved_stronger_or_eq_bb = 0L;
		//stronger_or_eq_bb[player][Strongest+1]=0L; state was already initialised
		//stronger_or_eq_bb[enemy][Strongest+1]=0L;
		// prev could be equal this
		for (int s = Strongest; s >= 0; s--) {// player move first
			stronger_or_eq_bb[s] ^= (moved_stronger_or_eq_bb ^= (move.from_bb[s]^move.to_bb[s]));
		}
		for (int s = 0; s <= Strongest; s++) {
			long temp_bb;
			if (move.from_bb[s]!=0) {
				temp_bb=move.from_bb[s];
				for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
					lsb_bb=Util.LastBit(temp_bb);
					int ind=Util.bit2ind(lsb_bb);
					piece_at[ind]=-1;
				}
				if ((temp_bb=move.from_bb[s] & move.enemy_from_bb)!=0) {
					player_bb[enemy] ^= temp_bb;
				}
				if ((temp_bb=move.from_bb[s] & ~move.enemy_from_bb)!=0) {
					player_bb[player] ^= temp_bb;
				}
			}
		}
		for (int s = 0; s <= Strongest; s++) {
			long temp_bb;
			if (move.to_bb[s]!=0) {
				if ((temp_bb=move.to_bb[s] & move.enemy_to_bb)!=0) {
					player_bb[enemy] ^= temp_bb;
					int piece=StrengthAndColor2InclPiece(s,enemy);
					for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
						lsb_bb=Util.LastBit(temp_bb);
						int ind=Util.bit2ind(lsb_bb);
						piece_at[ind]=piece;
					}
				}
				if ((temp_bb=move.to_bb[s] & ~move.enemy_to_bb)!=0) {
					player_bb[player] ^= temp_bb;
					int piece=StrengthAndColor2InclPiece(s,player);
					for(long lsb_bb;temp_bb!=0;temp_bb^=lsb_bb) {// removing moved pieces
						lsb_bb=Util.LastBit(temp_bb);
						int ind=Util.bit2ind(lsb_bb);
						piece_at[ind]=piece;
					}
				}
			}
		}
		empty_bb = ~(stronger_or_eq_bb[0]);
		// Update the non board stuff
		steps_remaining -= move.steps;
		zobr_hash ^= move.zobr_hash;

		if (steps_remaining == 0) {
			turn += player;
			player ^= 0x01;
			enemy ^= 0x01;
			steps_remaining = 4;
		}
		Material.debug=false;
		if ((move.info_mask & ArimaaMove.SETUP_MARK)!=0) {
			mat = Material.getMaterial(this);
		} else {
			try {
				mat = mat.capture(move.info_mask & ArimaaMove.CAPTURE_COUNTS);
			} catch (MaterialInconsistencyException b) {
				LogFile.writeln("Material inconsistency playing "+move+"in position "+this);
				b.logStackTrace();
			}
			if (Material.debug) {
				System.out.println(move);
				System.out.println(move.info_mask & ArimaaMove.CAPTURE_COUNTS);
				System.out.println(this);
			}
		}
		//if (mat != Material.getMaterial(this)) {
		//	LogFile.message("Material bug "+prev+move+"'"+move.move_ordering_value+"'"+this);
		//	Material.debug=true;
		//	LogFile.message("Prev Material:"+Material.getMaterial(prev));
		//	LogFile.message("this Material:"+Material.getMaterial(this));
		//	Material.debug=false;
		//}
		rand_score = ((int) (move.info_mask & ArimaaMove.RAND_SCORE_MASK)) >> 28;
	}

	public boolean existSlideMove(long p_start_bb, long p_dest_bb) {
		// expects compute_tertiary_bitboards was called already. Suicides are
		// handled.
		p_dest_bb &= empty_bb;
		p_start_bb &= touching_bb(p_dest_bb) & player_bb[player]
				& ~frozen_pieces_bb;
		// p_dest_bb &= touching_bb(p_start_bb); does not speed up

		if (p_start_bb == 0) {// no piece to move
			return false;
		}

		if ((p_start_bb & stronger_or_eq_bb[1]) != 0) {
			// unfrozen nonrabbit touching empty dest space
			return true;
		}
		// only rabbits remain to test
		p_start_bb &= (player == PL_GOLD) ? touching_esw_bb(p_dest_bb)
				: touching_wne_bb(p_dest_bb);
		return (p_start_bb != 0);
		// if there is unfrozen rabbit able to step to empty dest space return
		// true else return false
	}

	// Generates all 1 step slides (as moves) turn would be composed of several
	// moves (either slide's, or drags ... pulls, or pushes)
	public void genSlideMoves(MoveList result, long p_start_bb, long p_dest_bb) {
		// expects compute_tertiary_bitboards was called already. Suicides are
		// handled.
		p_dest_bb &= empty_bb;
		p_start_bb &= touching_bb(p_dest_bb) & player_bb[player] & ~frozen_pieces_bb;
		// p_dest_bb &= touching_bb(p_start_bb); does not speed up

		while (p_start_bb != 0) {
			long lsb_start_bb = Util.LastBit(p_start_bb);
			p_start_bb ^= lsb_start_bb;
			int pos_trap=0,pos_start=Util.bit2ind(lsb_start_bb);
			int slide_piece = getPieceType(pos_start);
			int p_slide_strength = slide_piece>>1;

			long to_bb = touching_bb(lsb_start_bb);
			long p_trap_bb = 0L;
			int p_trap_strength = 0; // default no suicide

			if ((lsb_start_bb & trap_protect_bb) != 0) { // suicide step away
				p_trap_bb = TRAP_SQUARES & to_bb;
				pos_trap = Util.bit2ind(p_trap_bb);
				p_trap_strength = getPieceType(pos_trap)>>1;
			}
			to_bb = p_dest_bb & can_step_bb(lsb_start_bb, slide_piece);
			while (to_bb != 0) {
				long lsb_to_bb = Util.LastBit(to_bb);
				to_bb ^= lsb_to_bb;

				ArimaaMove move = result.getMove();
				move.clear();

				// move.piece_bb[enemy][0] = lsb_to_bb; // slide move exception
				// remembering to_bb
				lsb_to_bb &= ~suicide_bb[player]; // suicide to unprotected trap
				move.from_bb[p_slide_strength] = lsb_start_bb;
				move.to_bb[p_slide_strength] = lsb_to_bb;
				move.from_bb[p_trap_strength] ^= p_trap_bb; 
				// does nothing if no suicide, (xor required for the same strength)

				move.zobr_hash ^= Zobr[p_slide_strength][player][pos_start];
				if (lsb_to_bb != 0) {
					move.zobr_hash ^= Zobr[p_slide_strength][player][Util.bit2ind(lsb_to_bb)];
				} else {
					move.info_mask += 1L << ((player << 5) + (p_slide_strength << 2));
				}
				if (p_trap_bb != 0) {
					move.zobr_hash ^= Zobr[p_trap_strength][player][pos_trap];
					move.info_mask += 1L << ((player << 5) + (p_trap_strength << 2));
				}
				if (player == PL_SILVER) {
					move.info_mask |= ArimaaMove.SILVER_MARK;
				}
				move.steps = 1;
			}
		}
	}

	public boolean existPullMove(long common_bb, long p_dest_bb, long e_start_bb) {
		// already filtered:
		// e_start_bb &= dominated_pieces_bb & player_bb[enemy];
		// common_bb &= touching_bb(e_start_bb) & player_bb[player] & stronger_or_eq_bb[1] &
		// ~frozen_pieces_bb &;

		p_dest_bb &= empty_bb;
		common_bb &= touching_bb(p_dest_bb);
		while (common_bb != 0L) {
			long lsb_p_start_bb = Util.LastBit(common_bb);
			common_bb ^= lsb_p_start_bb;
			int p_slide_strength = getPieceTypeOneSq(lsb_p_start_bb)>>1;

			long neighbour_bb = touching_bb(lsb_p_start_bb);
			long pull_e_start_bb = e_start_bb
					& neighbour_bb
					& ~stronger_or_eq_bb[p_slide_strength];
			// long pull_p_dest_bb = p_dest_bb & neighbour_bb;
			if (pull_e_start_bb != 0L) {// pull_p_dest_bb != 0L thanks to the common_bb filter
				return true;
			}
		}
		return false;
	}

	public void genPullMoves(MoveList result, long common_bb, long p_dest_bb,
			long e_start_bb) {
		// already filtered:
		// e_start_bb &= dominated_pieces_bb & player_bb[enemy];
		// common_bb &= touching_bb(e_start_bb) & player_bb[player] & stronger_or_eq_bb[1] &
		// ~frozen_pieces_bb &;

		p_dest_bb &= empty_bb;
		common_bb &= touching_bb(p_dest_bb);
		while (common_bb != 0L) {
			long lsb_p_start_bb = Util.LastBit(common_bb);
			common_bb ^= lsb_p_start_bb;
			int p_slide_strength = getPieceTypeOneSq(lsb_p_start_bb)>>1;

			long neighbour_bb = touching_bb(lsb_p_start_bb);
			long pull_e_start_bb = e_start_bb
					& neighbour_bb
					& ~stronger_or_eq_bb[p_slide_strength];
			long pull_p_dest_bb = p_dest_bb & neighbour_bb;
			if (pull_e_start_bb == 0L)
				continue; // pull_p_dest_bb != 0L thanks to common_bb filter

			long p_trap_bb = 0L;
			int p_trap_strength = 0; // default no player suicide
			if ((lsb_p_start_bb & trap_protect_bb) != 0L) { // suicide step away
				p_trap_bb = TRAP_SQUARES & neighbour_bb;
				p_trap_strength = getPieceTypeOneSq(p_trap_bb)>>1;
			}
			long lsb_e_dest_bb = lsb_p_start_bb & ~suicide_bb[enemy]; 
			// captured pulled piece would vanish

			while (pull_e_start_bb != 0L) {
				long lsb_e_start_bb = Util.LastBit(pull_e_start_bb);
				pull_e_start_bb ^= lsb_e_start_bb;
				int e_slide_strength = getPieceTypeOneSq(lsb_e_start_bb)>>1;

				long e_trap_bb = 0L;
				int e_trap_strength = 0; // default no opponent capture
				if ((lsb_e_start_bb & trap_protect_bb) != 0L) { // protection
																// pulled away
					e_trap_bb = TRAP_SQUARES & touching_bb(lsb_e_start_bb);
					e_trap_strength = getPieceTypeOneSq(e_trap_bb)>>1;
				}

				long tmp_p_dest_bb = pull_p_dest_bb;
				while (tmp_p_dest_bb != 0L) {
					long lsb_p_dest_bb = Util.LastBit(tmp_p_dest_bb);
					tmp_p_dest_bb ^= lsb_p_dest_bb;

					lsb_p_dest_bb &= ~suicide_bb[player]; // suicide to
															// unprotected trap
					ArimaaMove move = result.getMove();
					move.clear();
					move.from_bb[p_slide_strength] = lsb_p_start_bb;
					move.to_bb[p_slide_strength] = lsb_p_dest_bb;
					move.zobr_hash ^= Zobr[p_slide_strength][player][Util
							.FirstOne(lsb_p_start_bb)];
					if (lsb_p_dest_bb != 0L) {
						move.zobr_hash ^= Zobr[p_slide_strength][player][Util
								.FirstOne(lsb_p_dest_bb)];
					} else {
						move.info_mask += 1L << ((player << 5) + (p_slide_strength << 2));
					}
					if (p_trap_bb != 0L) {
						move.from_bb[p_trap_strength] ^= p_trap_bb; 
						// does nothing if no suicide, (xor required
						// for the same strength)
						move.zobr_hash ^= Zobr[p_trap_strength][player][Util
								.FirstOne(p_trap_bb)];
						move.info_mask += 1L << ((player << 5) + (p_trap_strength << 2));
					}
					move.from_bb[e_slide_strength] ^= move.enemy_from_bb = lsb_e_start_bb;
					move.to_bb[e_slide_strength] = move.enemy_to_bb = lsb_e_dest_bb;
					move.zobr_hash ^= Zobr[e_slide_strength][enemy][Util
							.FirstOne(lsb_e_start_bb)];
					if (lsb_e_dest_bb != 0L) {
						move.zobr_hash ^= Zobr[e_slide_strength][enemy][Util
								.FirstOne(lsb_e_dest_bb)];
					} else {
						move.info_mask += 1L << ((enemy << 5) + (e_slide_strength << 2));
					}
					if (e_trap_bb != 0L) {
						move.enemy_from_bb ^= e_trap_bb;
						move.from_bb[e_trap_strength] ^= e_trap_bb; 
						// does nothing if no suicide, (xor required
						// for the same strength)
						move.zobr_hash ^= Zobr[e_trap_strength][enemy][Util
								.FirstOne(e_trap_bb)];
						move.info_mask += 1L << ((enemy << 5) + (e_trap_strength << 2));
					}
					if (player == PL_SILVER) {
						move.info_mask |= ArimaaMove.SILVER_MARK;
					}
					move.steps = 2;
				}
			}
		}
	}

	public boolean existPushMove(long p_start_bb, long common_bb, long e_dest_bb) {
		// already done
		// e_start_bb &= dominated_pieces_bb;
		// p_start_bb &= touching_bb(common_bb) & player_bb[player] & stronger_or_eq_bb[1] &
		// ~frozen_pieces_bb;
		e_dest_bb &= empty_bb;
		common_bb &= touching_bb(e_dest_bb) & touching_bb(p_start_bb) & player_bb[enemy]; 
		// simillar to pulls with weaker enemy pulling

		while (common_bb != 0L) {
			long lsb_e_start_bb = Util.LastBit(common_bb);
			common_bb ^= lsb_e_start_bb;
			int e_slide_strength = getPieceTypeOneSq(lsb_e_start_bb)>>1;

			long neighbour_bb = touching_bb(lsb_e_start_bb);
			long push_p_start_bb = p_start_bb & neighbour_bb
					& stronger_or_eq_bb[e_slide_strength + 1];
			if (push_p_start_bb != 0L) {
				// push_e_dest_bb != 0L thanks to common_bb filter
				return true;
			}
		}
		return false;
	}

	public void genPushMoves(MoveList result, long p_start_bb, long common_bb,
			long e_dest_bb) {
		// already done
		// e_start_bb &= dominated_pieces_bb & player_bb[enemy];
		// p_start_bb &= touching_bb(common_bb) & player_bb[player] & stronger_or_eq_bb[1] &
		// ~frozen_pieces_bb;
		e_dest_bb &= empty_bb;
		common_bb &= touching_bb(e_dest_bb) & touching_bb(p_start_bb); 
		// simillar to pulls with weaker enemy pulling
		while (common_bb != 0L) {
			long lsb_e_start_bb = Util.LastBit(common_bb);
			common_bb ^= lsb_e_start_bb;
			int e_slide_strength = getPieceTypeOneSq(lsb_e_start_bb)>>1;

			long neighbour_bb = touching_bb(lsb_e_start_bb);
			long push_p_start_bb = p_start_bb & neighbour_bb
					& stronger_or_eq_bb[e_slide_strength + 1];
			if (push_p_start_bb == 0L)
				continue; // push_e_dest_bb != 0L guaranted by common_bb filter

			long push_e_dest_bb = e_dest_bb & neighbour_bb;
			long e_trap_bb = 0L;
			int e_trap_strength = 0; // default no opponent capture
			if ((lsb_e_start_bb & trap_protect_bb) != 0L) {
				// protected pushed away
				e_trap_bb = TRAP_SQUARES & neighbour_bb;
				e_trap_strength = getPieceTypeOneSq(e_trap_bb)>>1;
			}
			long lsb_p_dest_bb = lsb_e_start_bb & ~suicide_bb[player]; 
			// suicidal push to unprotected trap
			while (push_p_start_bb != 0L) {
				long lsb_p_start_bb = Util.LastBit(push_p_start_bb);
				push_p_start_bb ^= lsb_p_start_bb;
				int p_slide_strength = getPieceTypeOneSq(lsb_p_start_bb)>>1;

				long p_trap_bb = 0L;
				int p_trap_strength = 0; // default no player suicide
				if ((lsb_p_start_bb & trap_protect_bb) != 0L) { // suicide step
																// away
					p_trap_bb = TRAP_SQUARES & touching_bb(lsb_p_start_bb);
					p_trap_strength = getPieceTypeOneSq(p_trap_bb)>>1;
				}

				long tmp_e_dest_bb = push_e_dest_bb;
				while (tmp_e_dest_bb != 0L) {
					long lsb_e_dest_bb = Util.LastBit(tmp_e_dest_bb);
					tmp_e_dest_bb ^= lsb_e_dest_bb;

					lsb_e_dest_bb &= ~suicide_bb[enemy]; 
					// pushed to unprotected trap ... vanishes
					ArimaaMove move = result.getMove();
					move.clear();
					move.from_bb[p_slide_strength] = lsb_p_start_bb;
					move.to_bb[p_slide_strength] = lsb_p_dest_bb;
					move.zobr_hash ^= Zobr[p_slide_strength][player][Util
							.FirstOne(lsb_p_start_bb)];
					if (lsb_p_dest_bb != 0L) {
						move.zobr_hash ^= Zobr[p_slide_strength][player][Util
								.FirstOne(lsb_p_dest_bb)];
					} else {
						move.info_mask += 1L << ((player << 5) + (p_slide_strength << 2));
					}
					if (p_trap_bb != 0L) {
						move.from_bb[p_trap_strength] ^= p_trap_bb; 
						// does nothing if no suicide, (xor required for the same strength)
						move.zobr_hash ^= Zobr[p_trap_strength][player][Util
								.FirstOne(p_trap_bb)];
						move.info_mask += 1L << ((player << 5) + (p_trap_strength << 2));
					}
					move.from_bb[e_slide_strength] ^= move.enemy_from_bb = lsb_e_start_bb;
					move.to_bb[e_slide_strength] = move.enemy_to_bb = lsb_e_dest_bb;
					move.zobr_hash ^= Zobr[e_slide_strength][enemy][Util
							.FirstOne(lsb_e_start_bb)];
					if (lsb_e_dest_bb != 0L) {
						move.zobr_hash ^= Zobr[e_slide_strength][enemy][Util
								.FirstOne(lsb_e_dest_bb)];
					} else {
						move.info_mask += 1L << ((enemy << 5) + (e_slide_strength << 2));
					}
					if (e_trap_bb != 0L) {
						move.enemy_from_bb ^= e_trap_bb;
						move.from_bb[e_trap_strength] ^= e_trap_bb; 
						// does nothing if no suicide, (xor required for the same strength)
						move.zobr_hash ^= Zobr[e_trap_strength][enemy][Util
								.FirstOne(e_trap_bb)];
						move.info_mask += 1L << ((enemy << 5) + (e_trap_strength << 2));
					}
					if (player == PL_SILVER) {
						move.info_mask |= ArimaaMove.SILVER_MARK;
					}
					move.steps = 2;
				}
			}
		}
	}

	public boolean existDragMove(long p_start_bb, long p_dest_bb,
			long e_start_bb, long e_dest_bb) {
		if (steps_remaining < 2) {// no time for drag
			return false;
		}
		e_start_bb &= dominated_pieces_bb & player_bb[enemy];
		p_start_bb &= touching_bb(p_dest_bb) & touching_bb(e_start_bb)
				& player_bb[player] & stronger_or_eq_bb[1] & ~frozen_pieces_bb;

		if (existPullMove(p_start_bb & e_dest_bb, p_dest_bb, e_start_bb)) {
			return true;
		}
		if (existPushMove(p_start_bb, p_dest_bb & e_start_bb, e_dest_bb)) {
			return true;
		}
		return false;
	}

	/**
	 * Generates all push/pull moves from the current position. Piece captures
	 * are handled by this routine.
	 * 
	 * @param result
	 *            ArimaaMove[] to be extended
	 * @param p_start_bb
	 *            long Set of possible origin squares
	 * @param p_dest_bb
	 *            long Set of possible destination squares
	 * @param e_start_bb
	 *            long Set of possible enemy origin squares
	 * @param e_dest_bb
	 *            long Set of possible enemy destination squares
	 */
	public void genDragMoves(MoveList result, long p_start_bb, long p_dest_bb,
			long e_start_bb, long e_dest_bb) {

		if (steps_remaining < 2) {// no time for drag
			return;
		}

		e_start_bb &= dominated_pieces_bb & player_bb[enemy];
		p_start_bb &= touching_bb(p_dest_bb) & touching_bb(e_start_bb)
				& player_bb[player] & stronger_or_eq_bb[1] & ~frozen_pieces_bb;

		genPullMoves(result, p_start_bb & e_dest_bb, p_dest_bb, e_start_bb);
		genPushMoves(result, p_start_bb, p_dest_bb & e_start_bb, e_dest_bb);

	}

	public boolean existMove(long p_start_bb, long p_dest_bb) {
		// immobilization test?
		// Handle pass move
		if (steps_remaining != 4) {
			return true;
		}

		if (existSlideMove(p_start_bb, p_dest_bb))
			return true;
		if (existDragMove(p_start_bb, p_dest_bb, FULL_BB, FULL_BB))
			return true;
		return false;
	}

	/**
	 * Generates all valid slide, push, pull and pass moves for current position
	 * 
	 * @param result_steps
	 *            ArimaaMove[] to be extended
	 * @param p_start_bb
	 *            long
	 * @param p_dest_bb
	 *            long
	 */
	public void generateSteps(MoveList result_steps, long p_start_bb, long p_dest_bb) {

		// tertiary bitboards already caculated !!

		// Handle pass move
		if (steps_remaining != 4) {
			ArimaaMove move = result_steps.getMove();
			move.clear();
			move.steps = steps_remaining;
			move.info_mask = ArimaaMove.PASS_MARK|(player==PL_SILVER?ArimaaMove.SILVER_MARK:0);
		}

		// Handle piece moves
		genAllSteps(result_steps, p_start_bb, p_dest_bb, FULL_BB, FULL_BB);
	}

	/**
	 * Generates all basic moves from the current position (slide/push/pull).
	 * Piece captures are handled by this routine.
	 * 
	 * @param result_steps ArimaaMove[] to be extended
	 * @param p_start_bb long Set of possible origin squares
	 * @param p_dest_bb long Set of possible destination squares
	 * @param e_start_bb long Set of possible enemy origin squares
	 * @param e_dest_bb long Set of possible enemy destination squares
	 */

	public void genAllSteps(MoveList result_steps, long p_start_bb, long p_dest_bb,
			long e_start_bb, long e_dest_bb) {

		genSlideMoves(result_steps, p_start_bb, p_dest_bb);
		genDragMoves(result_steps, p_start_bb, p_dest_bb, e_start_bb, e_dest_bb);

	}

	/**
	 * Tests if an enemy piece has been captured, by comparing the current
	 * gamestate to the initial gamestate
	 * @param initial GameState
	 * @return boolean
	 */
	public boolean is_enemy_piece_captured(GameState initial) {
		return (this.mat.material_index[initial.enemy] != initial.mat.material_index[initial.enemy]);
	}

	/**
	 * Computes the mirrored index Reflects board about vertical axis
	 * @param index int
	 * @return int
	 */
	private int getMirrorIndex(int index) {
		assert ((index & ~0x3f) == 0);
		return index ^ 0x38;
	}

	/**
	 * Reflects position about vertical axis usage ??? Side to move remains the
	 * same
	 */

	public void mirror() {
		zobr_hash = 0L;
		long s_bb = 0L;
		long ms_bb = 0L;
		for (int s = Strongest; s >= 0; s--) {
			long temp_bb = stronger_or_eq_bb[s] ^ s_bb;
			s_bb ^= temp_bb;
			while (temp_bb != 0) {
				long lsb_bb = Util.LastBit(temp_bb);
				temp_bb ^= lsb_bb;
				int index = Util.FirstOne(lsb_bb);
				ms_bb |= 1L << getMirrorIndex(index);
				if ((player_bb[0]&lsb_bb)!=0) {
					zobr_hash ^= Zobr[s][0][index];
				} else {
					zobr_hash ^= Zobr[s][1][index];					
				}
			}
			stronger_or_eq_bb[s] = ms_bb;
		}
		player_bb[0]=Util.mirrorBB(player_bb[0]);
		player_bb[1]=Util.mirrorBB(player_bb[1]);
		empty_bb = ~stronger_or_eq_bb[0];
	}

	/**
	 * Computes the rotated index Rotates board 180 degrees
	 * 
	 * @param index int
	 * @return int
	 */
	private int getRotatedIndex(int index) {
		assert ((index & ~0x3f) == 0);
		return index & 0x3f;
	}

	/**
	 * Rotate the board 180 degrees usage ??? Changes side to move Swaps all
	 * piece colours
	 * 
	 * @param orig GameState
	 * 
	 */
	public void rotate() {
		turn += player;
		player ^= 0x01;
		enemy ^= 0x01;
		zobr_hash = 0L;
		long bb = 0L, r_bb = 0L;
		for (int s = Strongest; s >= 0; s--) {
			long temp_bb = stronger_or_eq_bb[s] ^ bb;
			bb ^= temp_bb;
			while (temp_bb != 0) {
				long lsb_bb = Util.LastBit(temp_bb);
				temp_bb ^= lsb_bb;
				int index = Util.FirstOne(lsb_bb);
				r_bb |= 1L << getRotatedIndex(index);
				if ((player_bb[0]&lsb_bb)!=0) {
					zobr_hash ^= Zobr[s][1][index];
				} else {
					zobr_hash ^= Zobr[s][0][index];
				}
			}
			stronger_or_eq_bb[s] = r_bb;
		}
		bb=Util.rotationBB(player_bb[0]);
		player_bb[0]=Util.rotationBB(player_bb[1]);
		player_bb[1]=bb;
		empty_bb = ~stronger_or_eq_bb[0];
	}

	public static void main(String args[]) {

		String text[] = {
				 "5w %13 +-----------------+%138|                 |%137|     D           |%136|     C           |%135|                 |%134|             R r |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
				 "5w %13 +-----------------+%138|                 |%137|   R D         r |%136| e   C           |%135|                 |%134|                 |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
				 "5b %13 +-----------------+%138|   r m           |%137|   R D           |%136| e   C           |%135|                 |%134|                 |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13", 
				"16w %13 +-----------------+%138| r r       r r r |%137| d d r r r       |%136| M e   E         |%135| R   h R h   m   |%134|                 |%133|   R   D R   H H |%132|   R       R     |%131|               R |%13 +-----------------+%13   a b c d e f g h%13"};

		String white = "1w Ee2 Me1 Hg2 Hb2 Df2 Dd1 Cc2 Cd2 Ra2 Rh2 Ra1 Rb1 Rc1 Rf1 Rg1 Rh1";
		String black = "1b ha7 db7 cc7 md7 ee7 cf7 dg7 hh7 ra8 rb8 rc8 rd8 re8 rf8 rg8 rh8";

		LogFile.writeln("new Gamestate");

		GameState gs = new GameState(white, black);
		System.out.println(gs.toBoardString());
		System.out.println(gs.toEPDString());
		System.out.println(gs.toSetupString());

		LogFile.writeln("Gamestate loop");
		for (String pos : text) {
			gs = new GameState(pos);
			System.out.println(gs.toBoardString());
			System.out.println(gs.toEPDString());
			System.out.println(gs.toSetupString());
		}
	}
}