package arimaa3;

import ai_util.*;

/**
 * Generates any one or two step moves that meet given criteria Also generates
 * moves that guarantee the piece is NOT FROZEN
 * 
 */

public class GenSteps extends ArimaaBaseClass {
	public GenSteps() {
	}

	// private static final boolean use_debug = false;

	// Working variables for gen_moves
	private MoveList step_list_stack = new MoveList(false);
	private MoveList doublestep_list_stack = new MoveList(false);

	/**
	 * Generate all 1 step slide moves. Piece froze status on dest square equals
	 * is_frozen Piece will not be self captured. It is OK to step to unprotected trap if
	 * is_frozen is true. Piece left unprotected on trap should vanish.
	 * Routine does not try to freeze piece by removing support ... 
	 */
	// TODO from_bb not marked ... is it needed? 
	public void genSlideMovesF(GameState game, MoveList step_list_result,
			long p_start_bb, long p_dest_bb, boolean is_frozen) {

		int player = game.player;
		int enemy = game.enemy;

		if (is_frozen) {
			p_dest_bb &= ~game.count_touch_bb[player][2];
		} else {
			// Remove bad trap squares, so suicide is not possible
			if ((p_start_bb & p_dest_bb & game.frozen_pieces_bb)!=0) {// try to unfreeze
				long unfroze_bb = touching_bb(p_start_bb & p_dest_bb & game.frozen_pieces_bb);
				game.genSlideMoves(step_list_result, FULL_BB, unfroze_bb);
			}
			p_dest_bb &= ~game.suicide_bb[player];
		}
		p_dest_bb &= game.empty_bb;
		p_start_bb &= game.player_bb[player] & ~game.frozen_pieces_bb;
		// Generate all non_rabbit slides at first
		for (int p_slide_strength = GameState.Strongest; p_slide_strength >= 0; p_slide_strength--) {
			long s_dest_bb = p_dest_bb;
			if (p_start_bb == 0) return;
			if (is_frozen) {
				s_dest_bb &= touching_bb(game.player_bb[enemy]&game.stronger_or_eq_bb[p_slide_strength + 1]);
			} else {
				s_dest_bb &= (~touching_bb(game.player_bb[enemy]&game.stronger_or_eq_bb[p_slide_strength + 1]))
						| game.count_touch_bb[player][2];
			}
			long s_start_bb = (game.stronger_or_eq_bb[p_slide_strength] ^ game.stronger_or_eq_bb[p_slide_strength + 1])
					& p_start_bb;
			p_start_bb ^= s_start_bb;
			s_start_bb &= ((p_slide_strength == 0) ? ((player == 0) ? touching_esw_bb(s_dest_bb)
							: touching_wne_bb(s_dest_bb))
							: touching_bb(s_dest_bb));

			while (s_start_bb != 0) {
				long lsb_start_bb = Util.LastBit(s_start_bb);
				s_start_bb ^= lsb_start_bb;

				long to_bb = touching_bb(lsb_start_bb);
				long p_trap_bb = 0L;
				int p_trap_strength = 0; // default no suicide thanks to p_trap_bb=0

				if ((lsb_start_bb & game.trap_protect_bb) != 0) {
					// suicide step away
					p_trap_bb = TRAP_SQUARES & to_bb;
					p_trap_strength = game.getPieceTypeOneSq(p_trap_bb)>>1;
				}

				to_bb &= s_dest_bb;
				if (p_slide_strength == 0) {
					to_bb &= (player == PL_GOLD) ? ~down_bb(lsb_start_bb)
							: ~up_bb(lsb_start_bb);
				}
				while (to_bb != 0) {
					long lsb_to_bb = Util.LastBit(to_bb);
					to_bb ^= lsb_to_bb;
					// Remove the place
					// Record the move
					ArimaaMove move = step_list_result.getMove();
					move.clear();
					move.from_bb[p_slide_strength] = lsb_start_bb;
					move.to_bb[p_slide_strength] = lsb_to_bb;
					// no suicide generated not generated for !is_frozen and 
					// suicide prevented by move reorder if is_frozen
					move.from_bb[p_trap_strength] ^= p_trap_bb; 
					// does nothing if no suicide, (xor required for
					// the same strength)
					move.info_mask |= ((player == PL_GOLD) ? 0
							: ArimaaMove.SILVER_MARK);

					move.zobr_hash = GameState.Zobr[p_slide_strength][player][Util
							.bit2ind(lsb_start_bb)];
					move.zobr_hash ^= GameState.Zobr[p_slide_strength][player][Util
							.bit2ind(lsb_to_bb)];
					if (p_trap_bb != 0) {
						move.zobr_hash ^= GameState.Zobr[p_trap_strength][player][Util
								.bit2ind(p_trap_bb)];
						move.info_mask += 1L << ((player << 5) + (p_trap_strength << 2));
					}
					move.steps = 1;
				}
			}
		}
	}

	/**
	 * Generate all one step moves from p_start_bb ending at p_dest_bb Piece is
	 * guaranteed not to be self-captured at p_dest_bb Piece could still be
	 * frozen at the destination. However other pieces may be captured in the
	 * process expects initial_position.compute_secondary_bitboards() was already
	 * called
	 * 
	 * @param position
	 *            GameState
	 * @param step_list_result
	 *            ArimaaMove[]
	 * @param p_start_bb
	 *            long
	 * @param p_dest_bb
	 *            long
	 */

	public void genOneStep(GameState position, MoveList step_list_result,
			long p_start_bb, long p_dest_bb) {
		position.genSlideMoves(step_list_result, p_start_bb, p_dest_bb
				& ~position.suicide_bb[position.player]);
	}

	/**
	 * Tests if one or two step moves to move a piece from p_start_bb to
	 * p_dest_bb exists. Piece is guaranteed not to be self-captured at
	 * p_dest_bb However other pieces may be captured in the process
	 * 
	 * @param position
	 *            GameState
	 * @param p_start_bb
	 *            long
	 * @param p_dest_bb
	 *            long
	 */
	public boolean existTwoStep(GameState position, long p_start_bb,
			long p_dest_bb) {

		int player = position.player;
		int enemy = position.enemy;
		long empty_bb = position.empty_bb;

		// Preconditions
		// initial_position.compute_tertiary_bitboards(); was already called
		long bad_trap_bb = position.suicide_bb[player];

		// Case 1: Unassisted 1 step slide or 2 step push
		// Remove bad trap squares
		long temp_dest_bb = p_dest_bb & ~bad_trap_bb;

		// Case 1a: Unassisted 1 step slide
		if (position.existSlideMove(p_start_bb, temp_dest_bb)) {
			return true;
		}

		// Case 1b: Unassisted 2 step push
		long e_start_bb = temp_dest_bb
				& position.player_bb[enemy]; // push needed
		if (e_start_bb != 0) {
			if (position.existDragMove(p_start_bb, e_start_bb,
					e_start_bb, FULL_BB)) {// pull moves would be skipped as
											// p_dest_bb & empty_bb = 0L
				return true;
			}
		}
		// Case 2: Unassisted 2 step slide
		long dest2_bb = p_dest_bb & empty_bb;
		long temp_start_bb = p_start_bb & ~position.frozen_pieces_bb;
		long inbetween_bb = touching_bb(p_start_bb) & touching_bb(dest2_bb)
				& ~bad_trap_bb & empty_bb; // could be frozen here !!

		// Start piece locations that have a hope of reaching
		temp_start_bb &= touching_bb(inbetween_bb);

		step_list_stack.clear();
		genSlideMovesF(position, step_list_stack, temp_start_bb,
				inbetween_bb, false);
		for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
			ArimaaMove move = step_list_stack.getCurrentMove();
			position.playMoveFull(move); 
			// secondary_bitboards computed
			// Setup for the second slide move
			long new_start_bb = move.getToBits();
			// suicide position not in inbetween_bb
			long illegal_dest_bb = move.getFromBits();
			long bad_trap_bb2 = position.suicide_bb[player];
			long td_bb = dest2_bb & ~illegal_dest_bb & ~bad_trap_bb2;

			//move_list_stack.levelUp();
			boolean found = position.existSlideMove(new_start_bb, td_bb);
			position.unplayMoveFull(move);
			if (found) {
				return true;
			}
			//move_list_stack.levelDn();
		}

		// Case 3: Assistance Required
		long temp_dest2_bb = touching_bb(p_start_bb) & p_dest_bb;
		long temp_start2_bb = touching_bb(p_dest_bb) & p_start_bb;

		// Case 3A: vacate destination square
		step_list_stack.clear();
		position.genSlideMoves(step_list_stack, temp_dest2_bb,
				FULL_BB);
		for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
			ArimaaMove move = step_list_stack.getCurrentMove();
			position.playMoveFull(move);

			// Setup for slide move
			long move_bb = move.getSetBits();
			long new_dest_bb = move_bb & ~empty_bb;
			// move to space created ... (the space created by a suicide is not touched by piece to move)
			long new_start_bb = ~move_bb & temp_start2_bb; // prevents oscilation

			//move_list_stack.levelUp();
			boolean found = position.existSlideMove(new_start_bb, new_dest_bb); 
			position.unplayMoveFull(move);
			if (found) {
				return true;
			}
			//move_list_stack.levelDn();
			
		}
		return false;
	}

	/**
	 * Generates all one or two step moves to move a piece from p_start_bb to
	 * p_dest_bb. Piece is guaranteed not to be self-captured at p_dest_bb
	 * However other pieces may be captured in the process
	 */
	public void genTwoStep(GameState position, MoveList doublestep_list_result,
			long p_start_bb, long p_dest_bb) {

		//LogFile.write("genTwoStep"+initial_position.toString());
		//LogFile.write("pstart "+bitboard(p_start_bb)+"pdest"+bitboard(p_dest_bb));
		
		int player = position.player;
		int enemy = position.enemy;
		long empty_bb = position.empty_bb;

		p_start_bb &= position.player_bb[player];
		//LogFile.write("pstart' "+bitboard(p_start_bb));
		// Preconditions
		// initial_position.compute_secondary_bitboards(); was already called
		long bad_trap_bb = position.suicide_bb[player];

		// Case 1: Unassisted 1 step slide or 2 step push
		// Remove bad trap squares
		long temp_dest_bb = p_dest_bb & ~bad_trap_bb;
		// Case 1a: Unassisted 1 step slide
		if (temp_dest_bb!=0) {
			//LogFile.write("genSlideMoves call pstart:"+bitboard(p_start_bb)+"dest:"+bitboard(temp_dest_bb));
			position.genSlideMoves(doublestep_list_result, p_start_bb, temp_dest_bb); 
			//LogFile.write("SlidesFound:"+doublestep_list_result.toString());
			// for empty squares (filtered by genSlideMoves)
			// Case 1b: Unassisted 2 step push
			long e_start_bb = temp_dest_bb
				& position.player_bb[enemy]; // push needed
			if (e_start_bb != 0) {
				//LogFile.write("genDragMoves call pstart:"+bitboard(p_start_bb)+"middle:"+bitboard(temp_dest_bb));
				position.genDragMoves(doublestep_list_result, p_start_bb, e_start_bb,
					e_start_bb, FULL_BB);
				//LogFile.write("DragsFound:"+doublestep_list_result.toString());
				// pull moves would be skipped as p_dest_bb & empty_bb = 0L
			}
		}
		// Case 2: Unassisted 2 step slide
		long dest2_bb = p_dest_bb & empty_bb;
		long temp_start_bb = p_start_bb & ~position.frozen_pieces_bb;
		long inbetween_bb = touching_bb(temp_start_bb) & touching_bb(dest2_bb)
				& ~bad_trap_bb & empty_bb; // could be frozen here !!

		// Start piece locations that have a hope of reaching
		temp_start_bb &= touching_bb(inbetween_bb);

		if (temp_start_bb != 0) {
			step_list_stack.clear();
			//LogFile.write("genSlideMovesF call pstart:"+bitboard(temp_start_bb)+"middle:"+bitboard(inbetween_bb));
			genSlideMovesF(position, step_list_stack, temp_start_bb,
				inbetween_bb, false);
			//LogFile.write("SlideFsFound:"+step_list_stack.toString());
			for (step_list_stack.levelFirst();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.write("Step_list_stack_loop:"+step_list_stack.toString());
				ArimaaMove move = step_list_stack.getCurrentMove();
				//LogFile.write("playing "+move.toString()+" in position "+initial_position.toString());
				position.playMoveFull(move); 
				//LogFile.write("got "+gs_stack[0].toString());
				// secondary_bitboards computed setup for the second slide move
				long new_start_bb = move.getToBits();
				// suicide position not in inbetween_bb
				long illegal_dest_bb = move.getFromBits();
				long bad_trap_bb2 = position.suicide_bb[player];
				long td_bb = dest2_bb & ~illegal_dest_bb & ~bad_trap_bb2;

				if (td_bb!=0) {
					step_list_stack.levelUp();
					//LogFile.write("genSlideMoves call on "+gs_stack[0].toString()+"newstart:"+bitboard(new_start_bb)+"dest:"+bitboard(td_bb));
					position.genSlideMoves(step_list_stack, new_start_bb, td_bb);
					//LogFile.write("SlidesFound:"+step_list_stack.toString());					
					for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
						ArimaaMove move2 = step_list_stack.getCurrentMove();
						// Combine the two moves and record the result
						doublestep_list_result.getMove().add(move, move2);
					}
					step_list_stack.levelDn();
				}
				position.unplayMoveFull(move);
			}
		}

		// Case 3: Assistance Required
		long temp_dest2_bb = touching_bb(p_start_bb) & p_dest_bb;
		long temp_start2_bb = touching_bb(p_dest_bb) & p_start_bb;

		// Case 3A: vacate destination square
		if (temp_dest2_bb!=0) {
			step_list_stack.clear();
			//LogFile.write("genSlideMoves vacate call pstart:"+bitboard(p_start_bb));
			position.genSlideMoves(step_list_stack, temp_dest2_bb,
				FULL_BB);
			//LogFile.write("SlidesFound:"+step_list_stack.toString());
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				ArimaaMove move = step_list_stack.getCurrentMove();
				position.playMoveFull(move);

				// Setup for slide move
				long move_bb = move.getSetBits();
				long new_dest_bb = move_bb & ~empty_bb;
				// move to space created (the space created by suicide is not touched by piece to move)
				long new_start_bb = ~move_bb & temp_start2_bb;
				// prevents oscilation

				if (new_start_bb!=0) {
					step_list_stack.levelUp();
					//LogFile.write("genSlideMoves call on "+gs_stack[0].toString()+"newstart:"+bitboard(new_start_bb)+"dest:"+bitboard(new_dest_bb));
					position.genSlideMoves(step_list_stack, new_start_bb,
							new_dest_bb);
					//LogFile.write("SlidesFound:"+step_list_stack.toString());					
					for (step_list_stack.levelFirst();step_list_stack.levelCont();step_list_stack.levelNext()) {
						ArimaaMove move2 = step_list_stack.getCurrentMove();
						// Combine the two moves and record the result
						doublestep_list_result.getMove().add(move, move2);
					}
					step_list_stack.levelDn();
				}
				position.unplayMoveFull(move);
			}
		}

		// Case 3B: support trap square
		// Figure out if a support move is even possible
		long supportable_trap_bb = bad_trap_bb & temp_dest2_bb;
		long temp_dest3_bb = touching_bb(supportable_trap_bb);

		if (temp_dest3_bb!=0) {
			step_list_stack.clear();
			//LogFile.write("genSlideMoves trap support call pdest:"+bitboard(temp_dest3_bb));
			position.genSlideMoves(step_list_stack, FULL_BB,
				temp_dest3_bb);
			//LogFile.write("SlidesFound:"+step_list_stack.toString());

			// Decided to allow all possible supporting moves

			for (step_list_stack.levelFirst();step_list_stack.levelCont();step_list_stack.levelNext()) {
				ArimaaMove move = step_list_stack.getCurrentMove();
				position.playMoveFull(move);

				// Setup for slide move
				long move_bb = move.getSetBits() & empty_bb;
				long new_dest_bb = touching_bb(move_bb) & TRAP_SQUARES;
				long new_start_bb = touching_bb(new_dest_bb) & p_start_bb;

				step_list_stack.levelUp();
				//LogFile.write("genSlideMoves call on "+gs_stack[0].toString()+"newstart:"+bitboard(new_start_bb)+"dest:"+bitboard(new_dest_bb));
				position.genSlideMoves(step_list_stack, new_start_bb,
					new_dest_bb);
				//LogFile.write("SlidesFound:"+step_list_stack.toString());					
				for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
					ArimaaMove move2 = step_list_stack.getCurrentMove();
					// Combine the two moves and record the result
					doublestep_list_result.getMove().add(move, move2);
				}
				step_list_stack.levelDn();
				position.unplayMoveFull(move);
			}
		}

		// Case 3C: unfreeze frozen p_start_bb piece
		long frozen_bb = temp_start2_bb & position.frozen_pieces_bb;
		long support_dest_bb = touching_bb(frozen_bb) & empty_bb;

		if (support_dest_bb!=0) {
			step_list_stack.clear();
			//LogFile.write("genSlideMoves unfreeze call pdest:"+bitboard(support_dest_bb));
			position.genSlideMoves(step_list_stack, FULL_BB,
				support_dest_bb);
			//LogFile.write("SlidesFound:"+step_list_stack.toString());

			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				ArimaaMove move = step_list_stack.getCurrentMove();
				position.playMoveFull(move);

				// Setup for slide move
				long move_bb = move.getSetBits() & empty_bb;
				long new_start_bb = touching_bb(move_bb) & frozen_bb;
				long new_dest_bb = p_dest_bb& ~bad_trap_bb; 
				// bad trap touching piece unfrozen by previous step remains bad ;

				if (new_dest_bb!=0) {
					step_list_stack.levelUp();
					//LogFile.write("genSlideMoves call on "+gs_stack[0].toString()+"newstart:"+bitboard(new_start_bb)+"dest:"+bitboard(new_dest_bb));
					position.genSlideMoves(step_list_stack, new_start_bb,
							new_dest_bb);
					//LogFile.write("SlidesFound:"+step_list_stack.toString());					
					for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
						ArimaaMove move2 = step_list_stack.getCurrentMove();
						// Combine the two moves and record the result
						doublestep_list_result.getMove().add(move, move2);
					}
					step_list_stack.levelDn();
				}
				position.unplayMoveFull(move);
			}
		}
	}

	/**
	 * Generates 1 step NF moves
	 */
	public void genOneStepNF(GameState initial_position, MoveList step_list_result,
			long p_start_bb, long p_dest_bb) {

		// initial_position.compute_secondary_bitboards(); was already called

		// Case 1: Slide the piece
		genSlideMovesF(initial_position, step_list_result, p_start_bb, p_dest_bb,
				false);

		// Case 2:
		// A valid p_start_bb piece is already on p_dest_bb but is frozen
		// Try to slide a player piece to unfreeze it

		genOneStep(initial_position, step_list_result, FULL_BB, touching_bb(p_start_bb
				& p_dest_bb & initial_position.frozen_pieces_bb));

	}

	/**
	 * Generates all two steps moves Target piece is not captured on dest square
	 * Target piece is not frozen on dest square
	 * 
	 * @param position
	 *            GameState
	 * @param doublestep_list_result
	 *            ArimaaMove[]
	 * @param p_start_bb
	 *            long
	 * @param p_dest_bb
	 *            long
	 */
	public void genTwoStepNF(GameState position, MoveList doublestep_list_result,
			long p_start_bb, long p_dest_bb) {

		// Case 1:
		// A valid p_start_bb piece is already on p_dest_bb but is frozen
		// Try to slide a player piece to unfreeze it

		// initial_position.compute_tertiary_bitboards(); already called
		p_start_bb &= position.player_bb[position.player];
		
		if ((p_start_bb==0)||(p_dest_bb==0)) return;

		long ori_frozen_bb = position.frozen_pieces_bb;
		long ori_player_bb = position.player_bb[position.player];
		long temp_bb = p_start_bb & p_dest_bb
				& position.frozen_pieces_bb;
		long help_dest_bb = touching_bb(temp_bb);
		if (help_dest_bb != 0) {
			genTwoStep(position, doublestep_list_result, FULL_BB, help_dest_bb);
		}
		// Case 2:
		// Use one step to move piece, other step for support piece, order does matter
		step_list_stack.clear(); //??
		genSlideMovesF(position, step_list_stack, p_start_bb,
				p_dest_bb, true);
		// should generate steps to unprotected trap as well without generating suicide :)
		for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
			ArimaaMove step1 = step_list_stack.getCurrentMove();
			position.playMoveFull(step1);

			// All squares touching the slide piece destination square
			long move_bb = step1.getSetBits(); // could have 2 or 3 bit's set
			// long slide_start_bb = move_bb & ~initial_position.empty_bb; //
			// could have 2 bits_set
			long slide_dest_bb = move_bb & position.empty_bb; // one bit set
			long slide_start_bb = move_bb ^ slide_dest_bb;
			if ((slide_start_bb & position.lone_defender_bb[position.player]) != 0) {
				slide_start_bb &= position.lone_defender_bb[position.player];
			}
			help_dest_bb = touching_bb(slide_dest_bb);

			if (help_dest_bb!=0) {
				step_list_stack.levelUp();
				genOneStep(position, step_list_stack, FULL_BB, help_dest_bb);
				for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
					ArimaaMove step2 = step_list_stack.getCurrentMove();
					// Combine the two moves and record the result
					doublestep_list_result.getMove().add(step1, step2);
				}
				step_list_stack.levelDn();
			}

			// Special case to consider
			// Sometimes helper piece has to slide first
			// h x
			//   R C d
			// For the Rabbit to get to x, unfrozen, the C has to move first
			// and it could be played even when x is trap!

			// Candidates must be
			// 1) touching start square
			// 2) Unfrozen before slide
			// 3) Frozen after slide

			long help_start_bb = touching_bb(slide_start_bb)
					& ~ori_frozen_bb & ori_player_bb
					& position.frozen_pieces_bb;
			// neighbours of the false (trap) bit of slide_start_bb does not
			// contain pieces to move

			if (help_start_bb != 0) {

				// 1) Generate the helper slide
				step_list_stack.levelUp();
				position.genSlideMoves(step_list_stack,
						help_start_bb, help_dest_bb);

				for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
					ArimaaMove step2reordered = step_list_stack.getCurrentMove();
					position.playMoveFull(step2reordered);

					// Generate the initial slide (May NOT be legal anymore!!!)
					step_list_stack.levelUp();
					position.genSlideMoves(step_list_stack, slide_start_bb, slide_dest_bb);
					for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
						ArimaaMove step1reordered = step_list_stack.getCurrentMove();
						// Combine the two moves and record the result
						doublestep_list_result.getMove().add(step1reordered, step2reordered); // TODO check the moves are ordered correctly
					}
					step_list_stack.levelDn();
					position.unplayMoveFull(step2reordered);
				}
				step_list_stack.levelDn();
			}
			position.unplayMoveFull(step1);
		}
		// Case 3:
		// Use both steps to move piece TODO 
		// For now call basic two step routine until it is fully debugged

		doublestep_list_stack.clear();
		genTwoStep(position, doublestep_list_stack, p_start_bb, p_dest_bb);
		// ^ calls this.repetition.increaseAge(); so must be called first if any
		for (doublestep_list_stack.levelNext();doublestep_list_stack.levelCont();doublestep_list_stack.levelNext()) {
			ArimaaMove doublestep = doublestep_list_stack.getCurrentMove();
			position.playMoveFull(doublestep);
			// For now, move is ok, if a non frozen piece is in the dest set
			// (even if the pushed piece counts ?)
			if ((~position.frozen_pieces_bb & p_dest_bb & doublestep.getSetBits()) != 0) {
				doublestep_list_result.getMove().copy(doublestep);
			}
			position.unplayMoveFull(doublestep);
		}
	}

	public static void main(String args[]) {

		String text[] = {
			"1|10w %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|     e   M       |%134|                 |%133|         R        |%132|        d       |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			, // Endgame Corner I Pos 4
			"0|2b %13 +-----------------+%138|     e           |%137|                 |%136|                 |%135|           E     |%134|                 |%133|       R         |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			, // Endgame Corner I Pos 4
			"1|2w %13 +-----------------+%138|     e           |%137|                 |%136|                 |%135|           E     |%134|                 |%133|       R         |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			, // Endgame Corner I Pos 4
			"1|3b %13 +-----------------+%138| r r r r r   r r |%137| M E d h         |%136| h D C c c   R H |%135| R R R R r m   e |%134|             H R |%133|                 |%132|         R       |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			,
			"5|4b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|         r   r   |%133| r   r   E   D r |%132| R M e R m     R |%131|   h R           |%13 +-----------------+%13   a b c d e f g h%13"
			,
			"4|5w %13 +-----------------+%138| r H r           |%137|   m E           |%136|   r R d c   r   |%135|   R   R R   R r |%134| R D e R   r C r |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			,
			"2|6b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|   C         M r |%133| R e r E         |%132|     D r r r   R |%131|     R R R R R   |%13 +-----------------+%13   a b c d e f g h%13"
			,
			"3|7b %13 +-----------------+%138|         r   r r |%137| r             c |%136|         D       |%135|   H   R         |%134|           M     |%133|   m           H |%132|   E e R       r |%131|             R R |%13 +-----------------+%13   a b c d e f g h%13"
			, // one step in reserve to step to another place on dest ...
			"2|8w %13 +-----------------+%138| r r     r   H   |%137|     c r R h   r |%136|   d       H R   |%135|             e   |%134|   h E       r C |%133|   d R c     R   |%132|     R           |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			,
			"5|9w %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|     e   M       |%134|                 |%133|   R   R         |%132|         d       |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			, // Endgame Corner I Pos 4
			"6|11w %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|     e   M       |%134|       H         |%133|       d          |%132|        R       |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			, // Endgame Corner I Pos 4
			"4|12w %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|           R     |%133|       d          |%132|        R C d   |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			, // Endgame Corner I Pos 4
			"3|13w %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|          R      |%133|       d          |%132|        R C     |%131|                 |%13 +-----------------+%13   a b c d e f g h%13"
			// Endgame Corner I Pos 4
		};

		// Run test positions
		for (String pos_text : text) {

			// Output the test position
			int pos_start = pos_text.indexOf("|")+1;
			int exp_num = Integer.parseInt(pos_text.substring(0,pos_start-1));
			pos_text = pos_text.substring(pos_start);
			GameState position = new GameState(pos_text);
			GenSteps test = new GenSteps();

			// Create the result array
			MoveList data = new MoveList(true);
			System.out.println(position);
			data.clear();
			test.genTwoStep(position, data, FULL_BB, RANK_3);
			//data.makeTopLevelUnique();
			if (data.levelSize()!=exp_num) {
				System.out.println("Bug report");
				System.out.println(position);
				System.out.println("Expected: "+exp_num+" gained: " + data.levelSize());
				System.out.println(data);
			} else {
				System.out.println("Test OK");
				System.out.println(position);
				System.out.println("Total moves: " + data.levelSize());
				System.out.println(data);
			} 
		}
	}
}
