package arimaa3;

import java.util.*;

import ai_util.*;

/**
 * Generates <b>all</b> possible moves for the given position
 * 
 * Plan to change it to work another way:
 * There should be granularity parameter and function LazyGenTurn which returns only (at least) granularity number of moves it returns false 
 * if there may be other moves and true if full turn generated
 * granularity is doubled by each call ususally starting with granularity 10 would be enough
 * Usage ... testing the goal in 1 cannot be prevented 
 * (at most cases first move prevents the goal so generating all moves would be big waste of time)
 * ... iterator on moves
 * .... this will be important only for no zugzwang detection
 * .... Goal prevention should work following way:
 * .... test "pass" opponents goal if exists, generate all goals, find their "counted goal path union" 
 * .... generate only turns affecting all the goal paths 
 * .... that prunning would not be easy as the first steps need not affect the paths
 */

// TODO the Evaluation cannot be invoked during the turn generation. It affects the repetition hash table due to capture testing.
// thanks to it multiple instances of the same move are generated (may be only if the last at most 2 rabbits exist).
// this is why mode 2 and mode 4 should be recodded
// mode 4 would only prepare step evaluation, but the evaluation would be done at special traversal
// mode 2 would generate whole turn and special traversal would chose the best move therefore mode 2 could be simulated by mode 0
// hmm I don't like the idea of the whole turn generation for the mode 2.
// I would rather use separate repetition hash table for the evaluation
// OK, genCaptures has two possible invocations now ... one prevents repetition, and the other prevents use of repetition hash table ;)
// test for goal (rather test for elimination) uses the variant allowing repetition, but preventing use of repetition hash table

public class GenTurn implements Constants {

	private TestForGoal goal = new TestForGoal();

	public GenTurn() {
	}

	// Working variables for gen_moves
	static private boolean debugging = false;
	private MoveList step_list_stack = new MoveList(false);
	private MoveList turn_list_data;// to make the global parameter accessible in subroutines
	private ArimaaMove[] turn_sums = new ArimaaMove[5]; //
	{
		for(int i = 0; i < 5; i++) {
			turn_sums[i] = new ArimaaMove();
		}
		turn_sums[0].clear(); // never changed
	}
	private RepetitionHashTable turn_repetition = new RepetitionHashTable();
	private HashSet<Long> positionForbidden;
	//private int mode=0; // 0 genTurn, 1 getNotation, 2 getBest, 3 checkTurnLegality, 4 genTurnEvaluated, 
	// 6 check winning, 8 genGoalThreats, 10 getBestandGoalThreats, 12 genFirstTurns
	//private boolean get_notation;
	//private boolean get_best=false;
	private ArimaaMove desired_turn = new ArimaaMove();
	private String notation_result = "";
	private boolean losingAllowed;
	private boolean revertibleAllowed;
	private ArimaaEvaluate eval;
	private ArimaaMove bestMove;
	private int betaScore=SCORE_INFINITY;;
	private int bestScore=-SCORE_MATE;
	private int toGenerate=0;// used for genFirstTurns 
	
	/** translates given turn ... should be possible
	 * in given position to standard notation ... should find way how to rearange pieces
	 * used for full turns only
	 * @param position
	 * @param turn
	 * @return
	 */
	public String getOfficialArimaaNotation(GameState position,
			ArimaaMove turn) {
		//mode = 1; // get_notation
		desired_turn.copy(turn);
		boolean found=false;
		// System.out.println("Desired: "+move);
		while (true) {
			try {
				this.turn_repetition.increaseAge();
				// age neednot be increased when increasing search deep as 
				// the position would be accepted with higher steps_remaining
				// Iterative deepening not to play stupid moves
				for (int steps=stepsRequired(turn);steps<=4;steps++) {
					//if (get_notation) {
					//	LogFile.message("Search for "+steps+" steps.............:");
					//}
					position.steps_remaining=steps;
					desired_turn.steps = steps;
					gen_turns1(position);
				}
			} catch (WinMoveFoundException b) {
				found=true;
				position.unplayMoveFull(turn);
			}
			position.steps_remaining=4;
			desired_turn.steps=4;
			assert (found);
			if (debugging) {
				ArimaaMove check =  new ArimaaMove(notation_result);
				if (!check.equals(turn)) {
					LogFile.message("Notation bug obtained "+notation_result+" and "+check+"#"+turn+found);
				} else {
					return notation_result;
				}
			} else {
				return notation_result;
			}
		}
	}

	private void gen_turns1(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					//	LogFile.message(move+"> initial");
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					//	LogFile.message(""+repeated_positions+move+"> repetition");
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Check if we're walking to desired_position
				if (turn_sums[step_list_stack.stack_ind].equalsUptoSteps(desired_turn)) {
					calculate_notation(turn_sums[step_list_stack.stack_ind].steps);
					throw new WinMoveFoundException();
				}
				if (!couldBeFinished(turn_sums[step_list_stack.stack_ind], desired_turn)) {
					//LogFile.message(move+"> cut " + dist_at_least);
					//dist_cuts++;
					position.unplayMove(step);
					continue;
				}
				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) < 4) {
					while (position.steps_remaining>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					//	LogFile.message(step+"> ...");
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					//	LogFile.message(step+"... <");
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	private void calculate_notation(int total_steps) {
		notation_result = "";
		int depth = 1, steps = 0;
		do {
			// System.out.println(position);
			ArimaaMove move = step_list_stack.move_list[step_list_stack.depth_ind[depth]];;
			//LogFile.message("calculate_notation ["+steps_available+"]"+move+position);
			// System.out.println(move);
			notation_result += move.SlideOrDragToOfficialArimaaNotation() + " ";
			depth ++; steps += move.steps;
			//position = depth_position[depth++];
		} while (steps < total_steps);
		notation_result = notation_result.trim();
	}
	
	/**
	 * Should find if winnig turn at given position exists
	 * used in TestForGoal as slow but safe method to compare with
	 * is fully replaced by TestForGoal in search
	 * another prunning is by revertible moves in at least 2 steps
	 * rather resign than play move losing in 1
	 * @param position
	 * @return
	 */
	public boolean checkWinningTurn(GameState position) {// just for TestForGoal testing
		gen_calls++;
		//mode = 6;// checkWinning
		turn_repetition.increaseAge();
		try {
			gen_turns6(position);
		} catch (WinMoveFoundException b) {
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			//LogFile.message("Winning move found leading to "+position);
			position.unplayMoveFull(moveFound);
			//LogFile.message("Original position "+position);
			//LogFile.message("Move "+moveFound);
			return true;
		}
		//LogFile.message("Winning move not found, final=?=original position "+position);
		return false;
	}
	
	private void gen_turns6(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Base case for recursion
				int sr;
				if ((sr=position.getStepsRemaining()) == 4) {
					if (position.isGameOver()) {
						if (position.getGameResult()>0) {
							throw new WinMoveFoundException();
						}
					}
				} else {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}
	/**
	 * Generates and evaluates turns, cut when evaluation exceeds beta. 
	 * @param position
	 * @param turn_list moves encountered which are goal threats (usefull especially when no cutting turn found)
	 * @param replyTurn if beta cut returns the cutting turn
	 * @param eval_ evaluator
	 * @param positionForbidden_ set of forbidden positions
	 * @param beta
	 * @return best score or first score better than beta
	 */
	public int getBestEvaluatedTurnAndThreats(GameState position, MoveList turn_list, ArimaaMove replyTurn, ArimaaEvaluate eval_, 
			HashSet<Long> positionForbidden_, int beta) {
		turn_list_data = turn_list;
		turn_list_data.levelClear();
		losingAllowed=false;
		revertibleAllowed=true;
		gen_calls++;
		//mode = 10;// getBest and goal threats
		turn_repetition.increaseAge();
		eval=eval_;
		positionForbidden=positionForbidden_;
		bestScore=-SCORE_MATE;
		betaScore=beta;
		bestMove=new ArimaaMove();
		try {
			gen_turns10(position);
			int ini_list_length=turn_list_data.range_stack[turn_list_data.stack_ind];
			for (turn_list_data.levelFirst();turn_list_data.depth_ind[turn_list_data.stack_ind]<=ini_list_length;turn_list_data.levelNext()) {
				ArimaaMove turn = turn_list_data.getCurrentMove();
				// filter to non-losing moves
				if (revertibleAllowed) {// we don't want revertible moves in any cases
					turn_list_data.levelReWrite();
					revertibleAllowed=false;
				}
				position.playMoveFull(turn);
				if (!turn.fasterRevertExists(position)) {
					turn_list_data.getMove().copy(turn);
				}
				position.unplayMove(turn);
			}
		} catch (WinMoveFoundException b) {
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			position.unplayMoveFull(moveFound);
			//LogFile.message("Winning move found!");
		}
		replyTurn.copy(bestMove);
		return bestScore;
	}

	private void gen_turns10(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) == 4) {
					int score;
					if (position.isGameOver()) {
						score=position.getGameResult();
						if (score>0) {
							bestScore=score;
							bestMove.copy(turn_sums[step_list_stack.stack_ind]);
							throw new WinMoveFoundException();
						}
					} else {
						score=-eval.Evaluate(position);
					}
					if (score > bestScore) {
						if (positionForbidden.contains(position.getPositionHash())) {
							score = -SCORE_MATE;
						}
						if (score > bestScore) {
							//score=-eval.Evaluate(new_position,true);// just for debugging
							bestMove.copy(turn_sums[step_list_stack.stack_ind]);
							bestScore=score; 
						}
						if (bestScore > betaScore) {
							throw new WinMoveFoundException();
						}
					}
					if ((goal.isGoalThreat(position)) && (score>-SCORE_MATE+1)) {
						ArimaaMove result = turn_list_data.getMove();
						result.copy(turn_sums[step_list_stack.stack_ind]);
					}
				} else {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}
	/**
	 * Returns best evaluated turn ... used in second last level of search to avoid creating the move list 
	 * just to traverse it for evaluation. The beta cut could prevent generation of the long list early.  
	 * genAllTurns should not generate moves losing in 1 when other moves exists
	 * another pruning is by revertible moves in at least 2 steps
	 * rather resign than play move losing in 1
	 * @param position
	 * @param replyTurn
	 * @param eval_
	 * @param positionForbidden_
	 * @param beta
	 * @return
	 */
	public int getBestEvaluatedTurn(GameState position, ArimaaMove replyTurn, ArimaaEvaluate eval_, 
			HashSet<Long> positionForbidden_, int beta) {
		gen_calls++;
		//mode = 2;// getBest
		turn_repetition.increaseAge();
		eval=eval_;
		positionForbidden=positionForbidden_;
		bestScore=-SCORE_MATE;
		betaScore=beta;
		bestMove=new ArimaaMove();
		try {
			gen_turns2(position);
		} catch (WinMoveFoundException b) {
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			position.unplayMoveFull(moveFound);
			//LogFile.message("Winning move found!");
		}
		replyTurn.copy(bestMove);
		return bestScore;
	}

	private void gen_turns2(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) == 4) {
					int score;
					if (position.isGameOver()) {
						score=position.getGameResult();
						if (score>0) {
							bestScore=score;
							bestMove.copy(turn_sums[step_list_stack.stack_ind]);
							throw new WinMoveFoundException();
						}
					} else {
						score=-eval.Evaluate(position);
					}
					if (score > bestScore) {
						if (positionForbidden.contains(position.getPositionHash())) {
							score = -SCORE_MATE;
						}
						if (score > bestScore) {
							//score=-eval.Evaluate(new_position,true);// just for debugging
							bestMove.copy(turn_sums[step_list_stack.stack_ind]);
							bestScore=score; 
						}
						if (bestScore > betaScore) {
							throw new WinMoveFoundException();
						}
					}
				} else {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/**
	 * Checks turn legality. Used for killer turns whose look promising both by availability and good score,
	 * but it has to be checked the turn could really be played in the position
	 * full turns only
	 * @param position
	 * @param turn
	 * @return
	 */
	public boolean isTurnLegal(GameState position,
			ArimaaMove turn) {
		//mode = 3; // legalityTest
		boolean found=false;

		// System.out.println("Desired: "+move);
		try {
			if (position.isKillerConsistent(turn)) {
				desired_turn.copy(turn);
				turn_repetition.increaseAge();
				// age neednot be increased when increasing search deep as 
				// the position would be accepted with higher steps_remaining
				//LogFile.message("isTurnLegal "+position+turn);
				for (int steps=stepsRequired(turn);steps<=4;steps++) {
					//LogFile.message("isTurnLegal steps"+steps);
					position.steps_remaining=steps;
					desired_turn.steps = steps;
					gen_turns3(position);
				}
			}
		} catch (WinMoveFoundException b) {
			found=true;
			position.unplayMoveFull(desired_turn);
		}
		position.steps_remaining=4;
		return found;
	}

	private void gen_turns3(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				if (turn_sums[step_list_stack.stack_ind].equalsUptoSteps(desired_turn)) {
					throw new WinMoveFoundException();
				}
				if (!couldBeFinished(turn_sums[step_list_stack.stack_ind], desired_turn)) {
					//dist_cuts++;
					position.unplayMove(step);
					continue;
				}
				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) < 4) {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/**
	 * Generates all non losing turns (generates all, but removes the losing ones)
	 * @param position
	 * @param turn_list
	 */
	public void genNonLosingTurns(GameState position, MoveList turn_list) {
		turn_list_data = turn_list;
		turn_list_data.levelClear();
		gen_calls++;
		//mode=0;// genTurn
		turn_repetition.increaseAge();
		losingAllowed=false;
		revertibleAllowed=true;
		// Generate the moves
		try {
			gen_turns0(position);
			int ini_list_length=turn_list_data.range_stack[turn_list_data.stack_ind];
			for (turn_list_data.levelFirst();turn_list_data.depth_ind[turn_list_data.stack_ind]<=ini_list_length;turn_list_data.levelNext()) {
				ArimaaMove turn = turn_list_data.getCurrentMove();
				// filter to non-losing moves
				position.playMoveFull(turn);
				total_non_losing_moves++;
				boolean revertible=turn.fasterRevertExists(position);
				if (!revertible) {
					total_non_revertible_moves++;
					if (revertibleAllowed) {
						turn_list_data.levelReWrite();
						revertibleAllowed=false;
					}
					turn_list_data.getMove().copy(turn);
				} else {
					revertible=turn.fasterRevertExists(position); 
					//TODO remove after debugging
					if (revertibleAllowed) {
					turn_list_data.getMove().copy(turn);
					}
				}
				position.unplayMove(turn);
			}
		} catch (WinMoveFoundException b) {// for the case there exists win in 1 move
			//LogFile.message("Winning move found!");
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			position.unplayMoveFull(moveFound);
		}
	}

	private void gen_turns0(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) == 4) {
					if (position.isGameOver()) {
						if (position.getGameResult()<0) { // Opponent won
							total_moves++;
						} else {
							//LogFile.message(new_position+"player win!?");
							turn_list_data.levelReWrite();
							total_moves++;
							total_non_losing_moves++;
							total_non_revertible_moves++;
							// Record the move
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							throw new WinMoveFoundException(); // we are happy with one winning move
						}
					} else {
						total_moves++;
						if (!goal.test(position)) {
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							if (goal.isGoalThreat(position)) {
								turn_list_data.makeMoveGoalThreat();
							} else if ((result.info_mask&ArimaaMove.PLAYER_CAPTURE_COUNTS[position.player])!=0) {
								turn_list_data.makeMoveCapture();// "fast sorting" for cut purposes
							}
						}
					}
				} else {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/**
	 * Returns true if more than sizeBound number of moves exist. Should not be called when immediate goal exist.
	 * @param position
	 * @param turn_list
	 * @param sizeBound
	 * @return
	 */
	public boolean genFirstTurns(GameState position, MoveList turn_list,int sizeBound) {
		turn_list_data = turn_list;
		turn_list_data.levelClear();
		gen_calls++;
		//mode=0x0c;// genTurn
		turn_repetition.increaseAge();
		toGenerate = sizeBound;
		losingAllowed=false;
		// Generate the moves
		try {
			gen_turns12(position);
		} catch (WinMoveFoundException b) {// we have generated enough moves ... for the case there exists win in 1 move shouold not be called
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			position.unplayMoveFull(moveFound);
			return true;
		}
		return false;
	}

	private void gen_turns12(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					//if (get_notation) {
					//	LogFile.message(move+"> initial");
					//}
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					//if (get_notation) {
					//	LogFile.message(""+repeated_positions+move+"> repetition");
					//}
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) == 4) {
					if (position.isGameOver()) {
						if (position.getGameResult()<0) { // Opponent won
							total_moves++;
						} else {
							//LogFile.message(new_position+"player win!?");
							turn_list_data.levelReWrite();
							total_moves++;
							total_non_losing_moves++;
							total_non_revertible_moves++;
							// Record the move
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							throw new WinMoveFoundException(); // we are happy with one winning move
						}
					} else {
						total_moves++;
						if (!goal.test(position)) {
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							if (toGenerate==0) {
								throw new WinMoveFoundException(); // we have already generated enough
							}
							toGenerate--;
							if (goal.isGoalThreat(position)) {
								turn_list_data.makeMoveGoalThreat();
							} else if ((result.info_mask&ArimaaMove.PLAYER_CAPTURE_COUNTS[position.player])!=0) {
								turn_list_data.makeMoveCapture();// "fast sorting" for cut purposes
							}
						}
					}
				} else {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					//if (get_notation) {
					//	LogFile.message(step+"> ...");
					//}
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					//if (get_notation) {
					//	LogFile.message(step+"... <");
					//}
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/** Should generate just goal threats to be used in goal quiesce search
	 * @param position
	 * @param turn_list
	 */
	public void genGoalThreatTurns(GameState position, MoveList turn_list) {
		turn_list_data = turn_list;
		turn_list_data.levelClear();
		gen_calls++;
		//mode=8;// genGoalThreats
		turn_repetition.increaseAge();
		losingAllowed=false;
		revertibleAllowed=true;
		// Generate the moves
		try {
			gen_turns8(position);
			int ini_list_length=turn_list_data.range_stack[turn_list_data.stack_ind];
			for (turn_list_data.levelFirst();turn_list_data.depth_ind[turn_list_data.stack_ind]<=ini_list_length;turn_list_data.levelNext()) {
				ArimaaMove turn = turn_list_data.getCurrentMove();
				// filter to non-losing moves
				if (revertibleAllowed) {// we don't want revertible moves in any cases
					turn_list_data.levelReWrite();
					revertibleAllowed=false;
				}
				position.playMoveFull(turn);
				if (!turn.fasterRevertExists(position)) {
					turn_list_data.getMove().copy(turn);
				}
				position.unplayMove(turn);
			}
		} catch (WinMoveFoundException b) {// for the case there exists win in 1 move
			//LogFile.message("Winning move found!");
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			position.unplayMoveFull(moveFound);
		}
		//LogFile.write(""+initial_position+turn_list_data.levelSize()+"threats");
	}

	private void gen_turns8(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}
				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}
				// Base case for recursion
				int sr;
				if ((sr=position.steps_remaining) == 4) {
					if (position.isGameOver()) {
						if (position.getGameResult()<0) { // Opponent won
							total_moves++;
						} else {
							//LogFile.message(new_position+"player win!?");
							turn_list_data.levelReWrite();
							total_moves++;
							total_non_losing_moves++;
							total_non_revertible_moves++;
							// Record the move
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							throw new WinMoveFoundException(); // we are happy with one winning move
						}
					} else {
						total_moves++;
						if (!goal.test(position)) {
							if (goal.isGoalThreat(position)) {
								ArimaaMove result = turn_list_data.getMove();
								result.copy(turn_sums[step_list_stack.stack_ind]);
							}
						}
					}
				} else {
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/**
	 * Generates all turns (losing are allowed when no other moves exist) used on first pass to
	 * get base ordering of root turns
	 * @param position
	 * @param turn_list
	 * @param eval_
	 */
	public void genAllTurnsEvaluated(GameState position, MoveList turn_list, ArimaaEvaluate eval_) {
		turn_list_data = turn_list;
		turn_list_data.levelClear();
		gen_calls++;
		//mode=4;// genTurn evaluated
		this.turn_repetition.increaseAge();
		losingAllowed=true;
		revertibleAllowed=true;
		eval = eval_;
		// Generate the moves
		try {
			gen_turns4(position);
			int ini_list_length=turn_list_data.range_stack[turn_list_data.stack_ind];
			for (turn_list_data.levelFirst();turn_list_data.depth_ind[turn_list_data.stack_ind]<=ini_list_length;turn_list_data.levelNext()) {
				ArimaaMove turn = turn_list_data.getCurrentMove();
				// filter to non-losing moves
				position.playMoveFull(turn);
				if (!goal.test(position)) {
					if (losingAllowed) {
						turn_list_data.levelReWrite();
						losingAllowed=false;
					}
					total_non_losing_moves++;
					boolean revertible=turn.fasterRevertExists(position);
					if (!revertible) {
						total_non_revertible_moves++;
						if (revertibleAllowed) {turn_list_data.levelReWrite();revertibleAllowed=false;}
						turn_list_data.getMove().copy(turn);
					} else if (revertibleAllowed) {
						turn_list_data.getMove().copy(turn);
					}
				}
				position.unplayMove(turn);
			}
		} catch (WinMoveFoundException b) {// for the case there exists win in 1 move
			ArimaaMove moveFound=turn_sums[step_list_stack.stack_ind];
			position.unplayMoveFull(moveFound);
			//LogFile.message("Winning move found!");
		}
	}

	private void gen_turns4(GameState position) throws WinMoveFoundException {

		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}
				// Base case for recursion
				if (position.getStepsRemaining() == 4) {
					if (position.isGameOver()) {
						if (position.getGameResult()<0) { // Opponent won
							total_moves++;
							if (losingAllowed) {
								// Record the move
								ArimaaMove result = turn_list_data.getMove();
								result.copy(turn_sums[step_list_stack.stack_ind]);
							}
						} else {
							//LogFile.message(new_position+"player win!?");
							turn_list_data.levelReWrite();
							losingAllowed=false;
							total_moves++;
							total_non_losing_moves++;
							total_non_revertible_moves++;
							// Record the move
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							throw new WinMoveFoundException(); // we are happy with one winning move
						}
					} else {
						total_moves++;
						boolean saveTurn=losingAllowed;
						if (!goal.test(position)) {
							if (losingAllowed) {
								turn_list_data.levelReWrite();
								losingAllowed=false;
							}
							saveTurn = true;
						}
						if (saveTurn) {
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
							if (step.info_mask == ArimaaMove.PASS_MARK) {
								result.move_ordering_value = eval.PassEvaluate(step.steps);
							}
							if (goal.isGoalThreat(position)) {
								result.move_ordering_value+=SCORE_THREAT;
							}
							result.move_ordering_value -= eval.Evaluate(position);
							// Best moves considered first. Best moves lead to worst positions
							//TODO add move evaluation ... initialised by 0 default, but negative by passes
							// (prefere 4 steppers to 3 steppers and much to 2 steppers and 1 steppers)
							// detect lemmings?
						}
					}
				} else {
					int sr=position.getStepsRemaining();
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/** lower bound for steps required for the turn 5 if not possible at all
	 * @param turn
	 * @return number of steps
	 */
	private int stepsRequired(ArimaaMove turn) {// we could expect it is turn in a position so <= 4
		int player_steps_required = 0, enemy_steps_required = 0, from_size, to_size, dist_steps; long from_bb, to_bb;
		if (turn.enemy_from_bb==0) {// optimized code for enemy not included
			boolean player_present=false;
			for(int s=0;s<=GameState.Strongest;s++) {
				if ((turn.from_bb[s])!=0) {
					player_present=true;
					from_size = Util.LowPopCnt(turn.from_bb[s]&~TRAP_SQUARES);
					to_size = Util.LowPopCnt(turn.to_bb[s]);
					dist_steps = Util.LowPopCnt(turn.to_bb[s]&~ArimaaBaseClass.touching_bb(turn.from_bb[s]));
					player_steps_required+=Math.max(from_size,to_size)+dist_steps;
				}
			}
			if (player_present && (player_steps_required==0)) {
				return 1;
			}
			return player_steps_required;
		}
		for(int s=0;s<=GameState.Strongest;s++) {
			if ((turn.from_bb[s])!=0) {
				if ((from_bb=(turn.from_bb[s] & turn.enemy_from_bb))!=0) {
					to_bb = turn.to_bb[s] & turn.enemy_to_bb;
					from_size = Util.LowPopCnt(from_bb&~TRAP_SQUARES);
					to_size = Util.LowPopCnt(to_bb);
					dist_steps = Util.LowPopCnt(to_bb&~ArimaaBaseClass.touching_bb(from_bb));
					enemy_steps_required+=Math.max(from_size,to_size)+dist_steps;
				}
				if ((from_bb=(turn.from_bb[s] & ~turn.enemy_from_bb))!=0) {
					to_bb = turn.to_bb[s] & ~turn.enemy_to_bb;
					from_size = Util.LowPopCnt(from_bb&~TRAP_SQUARES);
					to_size = Util.LowPopCnt(to_bb);
					dist_steps = Util.LowPopCnt(to_bb&~ArimaaBaseClass.touching_bb(from_bb));
					player_steps_required+=Math.max(from_size,to_size)+dist_steps;
				}
			}
		}
		enemy_steps_required=Math.max(enemy_steps_required,1); //we already know the enemy is present 
		player_steps_required=Math.max(player_steps_required,enemy_steps_required);
		//if ((player_steps_required+enemy_steps_required)==0) {
		//	LogFile.message("No steps required!" + this + turn);
		//}
		return player_steps_required+enemy_steps_required;
	}
	
	private boolean couldBeFinished(ArimaaMove prefix, ArimaaMove fullTurn) {
		int player_steps_required=0, enemy_steps_required=0, steps_allowed = fullTurn.steps- prefix.steps, from_size, to_size, dist_steps; long from_bb, to_bb, both_bb;
		if ((fullTurn.enemy_from_bb|prefix.enemy_from_bb)==0) {// optimized code for enemy not included
			for(int s=0;s<=GameState.Strongest;s++) {
				if ((prefix.from_bb[s]|fullTurn.from_bb[s])!=0) {
					from_bb = (prefix.to_bb[s]&~prefix.enemy_to_bb)^(fullTurn.from_bb[s]&~fullTurn.enemy_from_bb);
					to_bb = (prefix.from_bb[s]&~prefix.enemy_from_bb)^(fullTurn.to_bb[s]&~fullTurn.enemy_to_bb);
					both_bb = from_bb&to_bb;from_bb^= both_bb; to_bb^=both_bb;
					from_size = Util.LowPopCnt(from_bb);to_size = Util.LowPopCnt(to_bb);
					if (to_size>from_size) {
						return false;
					}
					from_size = Util.LowPopCnt(from_bb&~TRAP_SQUARES);
					dist_steps = Util.LowPopCnt(to_bb&~ArimaaBaseClass.touching_bb(from_bb));
					steps_allowed-=Math.max(from_size,to_size)+dist_steps;
					if (steps_allowed<0) {
						return false;
					}
				}
			}
			return steps_allowed>=0;
		}
		for(int s=0;s<=GameState.Strongest;s++) {
			if ((prefix.from_bb[s]|fullTurn.from_bb[s])!=0) {
				if ((((prefix.from_bb[s] & prefix.enemy_from_bb)|
					(fullTurn.from_bb[s] & fullTurn.enemy_from_bb)))!=0) {
					from_bb = (prefix.to_bb[s]&prefix.enemy_to_bb)^(fullTurn.from_bb[s]&fullTurn.enemy_from_bb);
					to_bb = (prefix.from_bb[s]&prefix.enemy_from_bb)^(fullTurn.to_bb[s]&fullTurn.enemy_to_bb);
					both_bb = from_bb&to_bb;from_bb^= both_bb; to_bb^=both_bb;
					from_size = Util.LowPopCnt(from_bb);to_size = Util.LowPopCnt(to_bb);
					if (to_size>from_size) {
						return false;
					}
					from_size = Util.LowPopCnt(from_bb&~TRAP_SQUARES);
					dist_steps = Util.LowPopCnt(to_bb&~ArimaaBaseClass.touching_bb(from_bb));
					enemy_steps_required+=Math.max(from_size,to_size)+dist_steps;
					if (enemy_steps_required+Math.max(enemy_steps_required,player_steps_required)>steps_allowed) {
						return false;
					}
				}
				if ((((prefix.from_bb[s] & ~prefix.enemy_from_bb)|
					(fullTurn.from_bb[s] & ~fullTurn.enemy_from_bb)))!=0) {
					from_bb = (prefix.to_bb[s]&~prefix.enemy_to_bb)^(fullTurn.from_bb[s]&~fullTurn.enemy_from_bb);
					to_bb = (prefix.from_bb[s]&~prefix.enemy_from_bb)^(fullTurn.to_bb[s]&~fullTurn.enemy_to_bb);
					both_bb = from_bb&to_bb;from_bb^= both_bb; to_bb^=both_bb;
					from_size = Util.LowPopCnt(from_bb);to_size = Util.LowPopCnt(to_bb);
					if (to_size>from_size) {
						return false;
					}
					from_size = Util.LowPopCnt(from_bb&~TRAP_SQUARES);
					dist_steps = Util.LowPopCnt(to_bb&~ArimaaBaseClass.touching_bb(from_bb));
					player_steps_required+=Math.max(from_size,to_size)+dist_steps;
					if (enemy_steps_required+player_steps_required>steps_allowed) {
						return false;
					}
				}
			}
		}
		return true;
	}
	/**
	 * internal worker function to gen moves base for separate modes
	 * gen_moves should not generate moves losing in 0 when other moves exists
	 * @param position
	 * @throws WinMoveFoundException
	 */
	private void gen_turns(GameState position) throws WinMoveFoundException {

		int mode=0;
		step_list_stack.clear();
		//position.compute_secondary_bitboards(); already called
		// Generate all legal slides/drags from current position
		position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
		while (true) {
			for (step_list_stack.levelNext();step_list_stack.levelCont();step_list_stack.levelNext()) {
				//LogFile.message("position "+position);
				ArimaaMove step = step_list_stack.getCurrentMove();
				// Play the step (slide/drag)
				turn_sums[step_list_stack.stack_ind].add(turn_sums[step_list_stack.stack_ind-1], step);
				position.playMoveFull(step);

				// Test for returning to initial position
				if (turn_sums[step_list_stack.stack_ind].isEmpty()) {
					//if (get_notation) {
					//	LogFile.message(move+"> initial");
					//}
					initial_repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				// Test for repetition of position
				// (test for repeated zorb_hash, with tiny probability false match
				// could be reported)
				long hash_code = position.getPositionHash();
				if (turn_repetition.isRepetition(hash_code)) { // Has side effects
					//if (get_notation) {
					//	LogFile.message(""+repeated_positions+move+"> repetition");
					//}
					repeated_positions++;
					position.unplayMove(step);
					continue;
				}

				if ((mode&1)==1) {
					// Check if we're walking to desired_position
					if (turn_sums[step_list_stack.stack_ind].equalsUptoSteps(desired_turn)) {
						if (mode==1) {// calculate notation
							calculate_notation(turn_sums[step_list_stack.stack_ind].steps);
						}
						throw new WinMoveFoundException();
					}
					if (!couldBeFinished(turn_sums[step_list_stack.stack_ind], desired_turn)) {
						//LogFile.message(move+"> cut " + dist_at_least);
						//dist_cuts++;
						position.unplayMove(step);
						continue;
					}
				}
				// Base case for recursion
				if (position.getStepsRemaining() == 4) {
					if ((mode|0xC)==0xC) {
						if (position.isGameOver()) {
							if (position.getGameResult()<0) { // Opponent won
								total_moves++;
								if (losingAllowed) {
									// Record the move
									ArimaaMove result = turn_list_data.getMove();
									result.copy(turn_sums[step_list_stack.stack_ind]);
								}
							} else {
								//LogFile.message(new_position+"player win!?");
								turn_list_data.levelReWrite();
								losingAllowed=false;
								total_moves++;
								total_non_losing_moves++;
								total_non_revertible_moves++;
								// Record the move
								ArimaaMove result = turn_list_data.getMove();
								result.copy(turn_sums[step_list_stack.stack_ind]);
								throw new WinMoveFoundException(); // we are happy with one winning move
							}
						} else {
							total_moves++;
							boolean saveTurn=losingAllowed;
							if (!goal.test(position)) {
								if (losingAllowed) {
									turn_list_data.levelReWrite();
									losingAllowed=false;
								}
								saveTurn = true;
							}
							if (saveTurn) {
								if (mode==8) {
									if (goal.isGoalThreat(position)) {
										ArimaaMove result = turn_list_data.getMove();
										result.copy(turn_sums[step_list_stack.stack_ind]);
									}				
								} else {
									ArimaaMove result = turn_list_data.getMove();
									result.copy(turn_sums[step_list_stack.stack_ind]);
									if (mode==0x0c) {
										if (toGenerate==0) {
											throw new WinMoveFoundException(); // we have already generated enough
										}
										toGenerate--;
									}					
									if (mode==4) {
										if (step.info_mask == ArimaaMove.PASS_MARK) {
											result.move_ordering_value = eval.PassEvaluate(step.steps);
										}
										if (goal.isGoalThreat(position)) {
											result.move_ordering_value+=SCORE_THREAT;
										}
										result.move_ordering_value -= eval.Evaluate(position);
										// Best moves considered first. Best moves lead to worst positions
										//TODO add move evaluation ... initialised by 0 default, but negative by passes
										// (prefere 4 steppers to 3 steppers and much to 2 steppers and 1 steppers)
										// detect lemmings?
									} else {
										if (goal.isGoalThreat(position)) {
											turn_list_data.makeMoveGoalThreat();
										} else if ((result.info_mask&ArimaaMove.PLAYER_CAPTURE_COUNTS[position.player])!=0) {
											turn_list_data.makeMoveCapture();// "fast sorting" for cut purposes
										}
									}
								}
							}
						}
					} else if (mode==10) {
						int score;
						if (position.isGameOver()) {
							score=position.getGameResult();
							if (score>0) {
								bestScore=score;
								bestMove.copy(turn_sums[step_list_stack.stack_ind]);
								throw new WinMoveFoundException();
							}
						} else {
							score=-eval.Evaluate(position);
						}
						if (score > bestScore) {
							if (positionForbidden.contains(position.getPositionHash())) {
								score = -SCORE_MATE;
							}
							if (score > bestScore) {
								//score=-eval.Evaluate(new_position,true);// just for debugging
								bestMove.copy(turn_sums[step_list_stack.stack_ind]);
								bestScore=score; 
							}
							if (bestScore > betaScore) {
								throw new WinMoveFoundException();
							}
						}
						if ((goal.isGoalThreat(position)) && (score>-SCORE_MATE+1)) {
							ArimaaMove result = turn_list_data.getMove();
							result.copy(turn_sums[step_list_stack.stack_ind]);
						}
					} else if (mode==2) {
						int score;
						if (position.isGameOver()) {
							score=position.getGameResult();
							if (score>0) {
								bestScore=score;
								bestMove.copy(turn_sums[step_list_stack.stack_ind]);
								throw new WinMoveFoundException();
							}
						} else {
							score=-eval.Evaluate(position);
						}
						if (score > bestScore) {
							if (positionForbidden.contains(position.getPositionHash())) {
								score = -SCORE_MATE;
							}
							if (score > bestScore) {
								//score=-eval.Evaluate(new_position,true);// just for debugging
								bestMove.copy(turn_sums[step_list_stack.stack_ind]);
								bestScore=score; 
							}
							if (bestScore > betaScore) {
								throw new WinMoveFoundException();
							}
						}
					} else if (mode==6) {
						if (position.isGameOver()) {
							if (position.getGameResult()>0) {
								throw new WinMoveFoundException();
							}
						}
					}
				} else {
					int sr=position.getStepsRemaining();
					while (position.getStepsRemaining()>1) {
						position.steps_remaining--;
						hash_code=position.getPositionHash();
						turn_repetition.isRepetition(hash_code);// blocking slower paths
					}
					// don't block full turns ... it would skip them
					position.steps_remaining=sr;// returning back
					// Generate the next steps (including pass)				
					//if (get_notation) {
					//	LogFile.message(step+"> ...");
					//}
					step_list_stack.levelUp();// level 5 never used for addressing
					position.generateSteps(step_list_stack, FULL_BB, FULL_BB);
					//if (get_notation) {
					//	LogFile.message(step+"... <");
					//}
					// step would be undone later after levelDn
					continue;
				}
				position.unplayMove(step);
			}
			step_list_stack.levelDn();
			// step to unplay should be refreshed
			ArimaaMove step_to_undo = step_list_stack.getCurrentMove();
			if (step_list_stack.stack_ind==0) {
				position.compute_secondary_bitboards();
				break;
			}
			position.unplayMove(step_to_undo);
		}
	}

	/**
	 * 
	 * Reports statistics on move generation
	 * 
	 * @return String Interesting statistics
	 */
	public String getStats() {
		String result = "";
		result += "GenTurn calls: " + gen_calls + "\n";
		result += "Total Moves: " + total_moves + "\n";
		result += "Total Non Losing Moves: " + total_non_losing_moves + "\n";
		result += "Total Non Revertible Moves: " + total_non_revertible_moves + "\n";
		result += "Initial Positions: " + initial_repeated_positions + "\n";
		result += "Repeated Positions: " + repeated_positions + "\n";
		result += "Dist Cuts: " + dist_cuts + "\n";
		return result;
	}

	/**
	 * Resets statistics
	 */

	public static void resetStats() {
		total_moves = 0;
		total_non_losing_moves = 0;
		total_non_revertible_moves = 0;
		repeated_positions = 0;
		initial_repeated_positions = 0;
		gen_calls = 0;
		dist_cuts = 0;
	}

	// Statistics collection stuff
	private static long total_moves = 0;
	private static long total_non_losing_moves = 0;
	private static long total_non_revertible_moves = 0;
	private static long repeated_positions = 0;
	private static long initial_repeated_positions = 0;
	private static long gen_calls = 0;
	private static long dist_cuts = 0;
	
	public static void main(String args[]) {
		
		debugging = true;
		String text[] = {
			//"13|16b %13 +-----------------+%138| r               |%137|                 |%136|                 |%135|                 |%134|                 |%133|                 |%132|                 |%131|               R |%13 +-----------------+%13   a b c d e f g h%13",
			"50|61w %13 +-----------------+%138| r H r           |%137|   m E           |%136|   r R d c   r   |%135|   R   R R   R r |%134| R D e R   r C r |%133|                 |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
			"48663|20w %13 +-----------------+%138| r r   m h   r   |%137| r   M     r   r |%136| r     E         |%135|   R           H |%134|               e |%133| D       C     R |%132|     H C   R     |%131| R R R       R R |%13 +-----------------+%13   a b c d e f g h%13",
			"125|16w %13 +-----------------+%138|     e           |%137| r               |%136|                 |%135|           E     |%134|                 |%133|       R         |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
			// Endgame Corner I Pos 4
			"88|16b %13 +-----------------+%138|     e           |%137| r               |%136|                 |%135|           E     |%134|                 |%133|       R         |%132|                 |%131|                 |%13 +-----------------+%13   a b c d e f g h%13", 
			// Endgame Corner I Pos 4
			"2293|26b %13 +-----------------+%138| r r r r r   r r |%137| M E d h         |%136| h D C c c   R H |%135| R R R R r m   e |%134|             H R |%133|                 |%132|         R       |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",
			"161|46b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|         r   r   |%133| r   r   E   D r |%132| R M e R m     R |%131|   h R           |%13 +-----------------+%13   a b c d e f g h%13",
			"131|58b %13 +-----------------+%138|                 |%137|                 |%136|                 |%135|                 |%134|   C         M r |%133| R e r E         |%132|     D r r r   R |%131|     R R R R R   |%13 +-----------------+%13   a b c d e f g h%13",
			"80|40b %13 +-----------------+%138|         r   r r |%137| r             c |%136|         D       |%135|   H   R         |%134|           M     |%133|   m           H |%132|   E e R       r |%131|             R R |%13 +-----------------+%13   a b c d e f g h%13",
			"2087|63w %13 +-----------------+%138| r r     r   H   |%137|     c r R h   r |%136|   d       H R   |%135|             e   |%134|   h E       r C |%133|   d R c     R   |%132|     R           |%131|                 |%13 +-----------------+%13   a b c d e f g h%13",//"336488|64w %13 +-----------------+%138|               r |%137|   E   M   C     |%136|                 |%135|     D   H   D   |%134|       H   C     |%133|   R             |%132|     R   R   R   |%131|   R   R   R   R |%13 +-----------------+%13   a b c d e f g h%13",
			"337916|65w %13 +-----------------+%138|               r |%137|   E   H   C     |%136|                 |%135|     D   H   D   |%134|       M   C     |%133|   R             |%132|     R   R   R   |%131|   R   R   R   R |%13 +-----------------+%13   a b c d e f g h%13",
			"337922|66w %13 +-----------------+%138|               r |%137|   C   H   E     |%136|                 |%135|     D   H   D   |%134|       M   C     |%133|   R             |%132|     R   R   R   |%131|   R   R   R   R |%13 +-----------------+%13   a b c d e f g h%13",
			"338030|68w %13 +-----------------+%138|               r |%137|   C   H   E     |%136|                 |%135|     D   M   D   |%134|       H   C     |%133|   R             |%132|     R   R   R   |%131|   R   R   R   R |%13 +-----------------+%13   a b c d e f g h%13",
			"338034|69w %13 +-----------------+%138|               r |%137|   C   H   E     |%136|                 |%135|     D   M   D   |%134|       C   H     |%133|   R             |%132|     R   R   R   |%131|   R   R   R   R |%13 +-----------------+%13   a b c d e f g h%13",
			"315908|71w %13 +-----------------+%138|               r |%137| E     H   C     |%136|                 |%135|     D   M   D   |%134|       C   H     |%133|   R             |%132|     R   R   R   |%131|   R   R   R   R |%13 +-----------------+%13   a b c d e f g h%13" 
		};

		// Run test positions
		for (String pos_text : text) {

			int pos_start = pos_text.indexOf("|")+1;
			int exp_num = Integer.parseInt(pos_text.substring(0,pos_start-1));
			pos_text = pos_text.substring(pos_start);

			// Output the test position
			GameState position = new GameState(pos_text);
			LogFile.message("Starting position:"+position);
			System.out.println(position);
			System.out.println(position.toSetupString());

			resetStats();
			long start_time = System.currentTimeMillis();
			GenTurn test = new GenTurn();
			MoveList result = new MoveList(true);
			test.genNonLosingTurns(position, result);
			boolean listAll=false;
			if (result.levelSize()!=exp_num) {
				LogFile.message("Bug report");
				LogFile.message(""+position);
				LogFile.message("Expected: "+exp_num+" gained: " + result.levelSize());
				listAll=true;
				//if (result.levelSize()<1000) listAll=true;
			} 
			try {
				position.compute_secondary_bitboards();
				test.gen_turns(position);
			} catch (WinMoveFoundException b) {// for the case there exists win in 1 move
				LogFile.message("Goal in 1 found at position "+position);
			}
			long elapsed_time = System.currentTimeMillis() - start_time;
			System.out.println("Elapsed time: " + elapsed_time + "ms");
			System.out.println(test.getStats());
			// Output ~10 resulting sample positions
			int count = 0; int sample = 1+(listAll?0:(result.levelSize()/10));
			for (result.levelFirst();result.levelCont();result.levelNext()) {
				ArimaaMove turn = result.getCurrentMove();
				if (count % sample == 0) {
					LogFile.message(count 
						+ test.getOfficialArimaaNotation(position, turn) + " " + turn);
				}
				count++;
			}
			System.out.println(test.getStats());
		}
	}
}