package arimaa3;

import ai_util.*;

//import java.util.*;

/**
 * Support for material evaluation not to compute it for each position, position just contains material object
 * the object knows it's own material evaluation
 * @author hroch
 *
 */
public class Material extends ArimaaBaseClass {

	// each material generated at most once so no need to compress this
	private static final int MaterialOverFull = 9 * 3 * 3 * 3 * 2 * 2;
	private static final int MaterialIndexShift[] = { 1, 9, 9 * 3, 9 * 3 * 3,
			9 * 3 * 3 * 3, 9 * 3 * 3 * 3 * 2, 9 * 3 * 3 * 3 * 2 * 2 };
	//private static final int HameWeights[][] = { 8.10.2012 divided by 5/2
	//		{ 1065, 105, 20, 0, 0, 0, 0, 0 }, 
	//		{ 105, 270, 65, 10, 0, 0, 0, 0 },
	//		{ 20, 65, 200, 35, 5, 0, 0, 0 }, 
	//		{ 0, 10, 35, 135, 20, 5, 0, 0 },
	//		{ 0, 0, 5, 20, 90, 15, 0, 0 }, 
	//		{ 0, 0, 0, 5, 15, 60, 10, 0 },
	//		{ 0, 0, 0, 0, 0, 10, 35, 5 }, 
	//		{ 0, 0, 0, 0, 0, 0, 5, 30 } };

	private static final int HameWeights[][] = {
		{ 426, 42,  8,  0,  0,  0,  0,  0 }, 
		{  42,108, 26,  4,  0,  0,  0,  0 },
		{   8, 26, 80, 14,  2,  0,  0,  0 }, 
		{   0,  4, 14, 54,  8,  2,  0,  0 },
		{   0,  0,  2,  8, 36,  6,  0,  0 }, 
		{   0,  0,  0,  2,  6, 24,  4,  0 },
		{   0,  0,  0,  0,  0,  4, 14,  2 }, 
		{   0,  0,  0,  0,  0,  0,  2, 12 }};

	private static Material MaterialTable[][] = new Material[MaterialOverFull][MaterialOverFull];
	/**
	 * translation from index to number of pieces of corresponding strength
	 */
	public static int PieceCount[][] = new int[MaterialOverFull][6]; 
	
	static {// "hash" table for used materials
		for (int index = 0; index < MaterialOverFull; index++) {
			for (int s = 0; s < 6; s++) {
				PieceCount[index][s] = (index % MaterialIndexShift[s + 1])
						/ MaterialIndexShift[s];
			}
		}		
		//long start_time = System.currentTimeMillis();
		//LogFile.write("Material setup started at "+start_time);

		int index[]=new int[2];
		for (index[0] = 0; index[0] < MaterialOverFull; index[0]++) {
			for (index[1] = 0; index[1] < MaterialOverFull; index[1]++) {
				MaterialTable[index[0]][index[1]] = new Material(index);
			}
		}
		//long end_time = System.currentTimeMillis();
		//long elapsed_time = end_time - start_time;
		//LogFile.write("Material setup ended at "+end_time);
		//LogFile.write("Material setup lasted "+elapsed_time);		
	}

	/** debug mode?
	 * 
	 */
	public static boolean debug=false;
	
	/** current material state indexed by color
	 */
	public int material_index[] = new int[2];
	/** was HAME already computed? It is computed once per program invocation.
	 */
	public boolean delayed=true;
	private int HAME; // material evaluation

	/** Constructor should know it's own material state (index)
	 */
	private Material(int index[]) {
		for (int c = 0; c < 2; c++) {
			material_index[c] = index[c];
		}
		//HAME = HAMEcalc(); delayed = true instead ... trying to prevent long init
	}

	/** 
	 * @return HAME evaluation ... computes it if not computed yet
	 */
	public int getHAME() {
		if (delayed) {
			HAME = HAMEcalc();
			delayed=false;
		}
		return HAME;
	}
	
	/**
	 * Computes material for given position. During the game the material is just updated,
	 * but for invocation in the middle of the game this routine is needed. 
	 * It does not check given position is legal so having 5 dogs need not cause problems here.
	 * The problems could appear later after captures.
	 * @param position
	 * @return material corresponding to the position
	 */
	public static Material getMaterial(GameState position) {
		int mindex[] = new int[2];
		for (int c = 0; c < 2; c++) {
			mindex[c] = 0;
			for (int s = 0; s <= GameState.Strongest; s++) {
				long piece_bb = position.player_bb[c] & (position.stronger_or_eq_bb[s]
					^ position.stronger_or_eq_bb[s + 1]);
				int piece_cnt = (s == 0) ? 
					Util.PopCnt(piece_bb) : 
					Util.LowPopCnt(piece_bb);
					if (debug) {
						LogFile.message("Mat["+c+","+s+"]="+piece_cnt+"/"+Long.toHexString(piece_bb));
					}
				mindex[c] += piece_cnt * MaterialIndexShift[s];
				if (s < GameState.Strongest) {
					if (piece_cnt * MaterialIndexShift[s]>=MaterialIndexShift[s+1]) {
						System.out.println("too many pieces of strength "+s+"("+piece_cnt+")");
					}
				}
			}
		}
		if ((mindex[0]>=MaterialOverFull)||(mindex[1]>=MaterialOverFull)) {
			System.out.println(position+"OverFull Material "+mindex[0]+": "+mindex[1]);
			return null;
		}
		if (debug) {
			LogFile.message("Strongest "+GameState.Strongest + GameState.InclPieceNames + " Material ["+mindex[0]+','+mindex[1]+"]");
			LogFile.message("Length " + position.stronger_or_eq_bb.length + position);
		}
		//if (MaterialTable[mindex[0]][mindex[1]] == null) {
		//	MaterialTable[mindex[0]][mindex[1]] = new Material(mindex);
		//}
		return MaterialTable[mindex[0]][mindex[1]];
	}

	/**
	 * Does not check there is full set of given stones in the position
	 * @param player 
	 * @param strength
	 * @return material corresponding to undo of capture piece of given strength and player
	 * @throws MaterialInconsistencyException
	 */
	public Material uncapture(int player, int strength) throws MaterialInconsistencyException {
		int mindex[] = new int[2];
		// there could be check added not to have 2 camels or so
		mindex[player] = material_index[player] + MaterialIndexShift[strength];
		mindex[player ^ 1] = material_index[player ^ 1];
		if ((mindex[0]<0) || mindex[0]>=MaterialOverFull) {
			System.out.println("Invalid material 0:"+mindex[0]+PieceCount[material_index[0]]);
			//System.out.println(TabletoString());
			throw new MaterialInconsistencyException();
		} 
		if ((mindex[1]<0) || mindex[1]>=MaterialOverFull) {
			System.out.println("Invalid material 1:"+mindex[1]+PieceCount[material_index[1]]);
			//System.out.println(TabletoString());
			throw new MaterialInconsistencyException();
		} 
		//if (MaterialTable[mindex[0]][mindex[1]] == null) {
		//	MaterialTable[mindex[0]][mindex[1]] = new Material(mindex);
		//}
		return MaterialTable[mindex[0]][mindex[1]];
	}

	/**
	 * @param player 
	 * @param strength
	 * @return material corresponding to capture piece of given strength and player
	 * @throws MaterialInconsistencyException
	 */
	public Material capture(int player, int strength) throws MaterialInconsistencyException {
		int mindex[] = new int[2];
		if(PieceCount[material_index[player]][strength]==0) {
			System.out.println("Error in capture ... not enough material! (player "+player+" strength "+strength+")");
			//System.out.println(TabletoString());
			throw new MaterialInconsistencyException();
		}
		mindex[player] = material_index[player] - MaterialIndexShift[strength];
		mindex[player ^ 1] = material_index[player ^ 1];
		if ((mindex[0]<0) || mindex[0]>=MaterialOverFull) {
			System.out.println("Invalid material 0:"+mindex[0]+PieceCount[material_index[0]]);
			//System.out.println(TabletoString());
			throw new MaterialInconsistencyException();
		} 
		if ((mindex[1]<0) || mindex[1]>=MaterialOverFull) {
			System.out.println("Invalid material 1:"+mindex[1]+PieceCount[material_index[1]]);
			//System.out.println(TabletoString());
			throw new MaterialInconsistencyException();
		} 
		//if (MaterialTable[mindex[0]][mindex[1]] == null) {
		//	MaterialTable[mindex[0]][mindex[1]] = new Material(mindex);
		//}
		return MaterialTable[mindex[0]][mindex[1]];
	}

	/** 
	 * @param piece
	 * @return material object corresponding to capture of given piece starting with this material
	 * @throws MaterialInconsistencyException
	 */
	public Material capture_piece(int piece) throws MaterialInconsistencyException  {
		return capture(piece&1, piece>>1);
	}
	
	/** 
	 * @param capture_mask is sum of count << ((strength<<2)+(player<<5)) values
	 * @return material object corresponding to undo of capture of set of pieces given by the capture mask starting with this material
	 * @throws MaterialInconsistencyException
	 */
	public Material uncapture(long capture_mask) throws MaterialInconsistencyException {
		Material mat = this;
		while (capture_mask != 0) {
			long lsb_mask = Util.LastBit(capture_mask);
			capture_mask ^= lsb_mask;
			int index = Util.FirstOne(lsb_mask);
			int player = index >>> 5;
			int strength = (index & 0x1f) >>> 2;
			int count = 1 << (index & 0x03);
			while (count > 0) {
				//if (mat.captures[player][strength] == null) {
				//	mat.captures[player][strength] = mat.capture(player,
				//			strength);
				//}
				mat = mat.uncapture(player,strength);
				count--;
			}
		}
		return mat;
	}

	/** 
	 * @param capture_mask is sum of count << ((strength<<2)+(player<<5)) values
	 * @return material object corresponding to capture of set of pieces given by the capture mask starting with this material
	 * @throws MaterialInconsistencyException
	 */
	public Material capture(long capture_mask) throws MaterialInconsistencyException {
		Material mat = this;
		while (capture_mask != 0) {
			long lsb_mask = Util.LastBit(capture_mask);
			capture_mask ^= lsb_mask;
			int index = Util.FirstOne(lsb_mask);
			int player = index >>> 5;
			int strength = (index & 0x1f) >>> 2;
			int count = 1 << (index & 0x03);
			while (count > 0) {
				//if (mat.captures[player][strength] == null) {
				//	mat.captures[player][strength] = mat.capture(player,
				//			strength);
				//}
				mat = mat.capture(player,strength);
				count--;
			}
		}
		return mat;
	}

	public String toString() {
		String result = "";
		for(int c=0;c<2;c++) {
			result+=material_index[c]+" ";
			for(int s=0;s<6;s++) {
				int mi=material_index[c];
				if (PieceCount[mi][s]!=0) {
					result+=PieceCount[mi][s];
					if (s<=GameState.Strongest) {
						result+=GameState.StrengthAndColor2Name(s, c);
					} else {
						result+="("+(s-GameState.Strongest)+") over!";
					}
				}
			}
			result+=":";
		}
		result+=HAME;
		return result;
	}
	
	/** 
	 * @return the HAME evaluation for the material
	 */
	private int HAMEcalc() {
		int powers[][] = new int[2][8];
		int lastNr[] = new int[2];
		int tot_pieces[] = new int[2];
		int resist[] = new int[2];
		for (int c = 0; c < 2; c++) {
			int k = 0,mi=material_index[c];
			for (int s = 5; s > 0; s--) {
				for (int j = 0; j < PieceCount[mi][s]; j++) {
					powers[c][k++] = s;
				}
			}
			lastNr[c] = k;
			for (; k < 8;)
				powers[c][k++] = 0;
			tot_pieces[c] = lastNr[c] + PieceCount[mi][0];
			resist[c] = tot_pieces[c] + lastNr[c];
			// nonrabbit is considered twice as good blocker than rabbit
		}
		int maxLastNr = Math.max(lastNr[0], lastNr[1]);
		int score = 0;
		// drag nr score max 2475
		for (int p = 0; p < maxLastNr; p++) {
			for (int pp = 0; pp < maxLastNr; pp++) {
				if (powers[0][p] > powers[1][pp])
					score += HameWeights[p][pp];
				if (powers[1][p] > powers[0][pp])
					score -= HameWeights[p][pp];
			}
		}
		// bigger army score max 9000 usually ~ 60
		// 6.5.2012 600->1200 factor changed
		if (tot_pieces[0] > tot_pieces[1]) {
			if (resist[1] > 0) {// total elimination does not need score :)
				score += (tot_pieces[0] - tot_pieces[1]) * 1200 / resist[1];
			}
		} else {
			if (resist[0] > 0) {// total elimination does not need score :)
				score += (tot_pieces[0] - tot_pieces[1]) * 1200 / resist[0];
			}
		}

		// piece diversibility adjustment max 5 :)
		for (int p = 1; p < 6; p++) {
			if (PieceCount[material_index[0]][p] > 0)
				score++;
			if (PieceCount[material_index[1]][p] > 0)
				score--;
		}

		// goaling score max 9985
		if (PieceCount[material_index[0]][0] > 0)
			score += 10000 - 420 / PieceCount[material_index[0]][0];
		if (PieceCount[material_index[1]][0] > 0)
			score -= 10000 - 420 / PieceCount[material_index[1]][0];

		// 420, 210, 140, 105, 84, 70, 60, 52
		// difference 210, 70, 35, 21, 14, 10, 8

		return score; // max is bigger than 30000
	}

	/**
	 * String representation of all capture possibilities for all material states but it lasts forever ... just log it instead
	 */
	private static void LogTheTable() {
		String result = "MaterialTable\n";
		for(int mg=MaterialOverFull-1;mg>=0;mg--) {
			for(int ms=MaterialOverFull-1;ms>=0;ms--) {
				Material mat=MaterialTable[mg][ms];
				if (mat==null) continue;
				result += "("+mg+","+ms+")=("+mat.material_index[0]+","+mat.material_index[1]+")";
				for(int c=0;c<2;c++) {
					result+="[";
					for(int s=0;s<5;s++) {
						result += PieceCount[mat.material_index[c]][s]+",";
					}
					result+=PieceCount[mat.material_index[c]][5]+"]";
				}
				result+="["+mat.getHAME()+"]";
				LogFile.message(result); result="";
				for(int c=0;c<2;c++) {
					for(int s=0;s<6;s++) {
						if (PieceCount[mat.material_index[c]][s]>0) {
							try {
								Material cap_to = MaterialTable[mg][ms].capture(c,s);
								result+=" ("+c+":"+s+")->("+cap_to.material_index[0]+","+cap_to.material_index[1]+")["+cap_to.getHAME()+"]\n";
							} catch (MaterialInconsistencyException b) {// for the case there exists win in 1 move
								System.out.print("Bug in material table2string. That should never appear!");
							}
						}
					}
				}
			}
		}
		LogFile.message(result);
	}	

	public static void main(String args[]) {
		LogTheTable();
	}
}